/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "utils.h"

#include "common.h"
#include "extension.h"
#include "opensslexception.h"
#include "version.h"

#include <QtCore/QByteArray>
#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QStringList>

#include <QtNetwork/QSslKey>

#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/bn.h>
#include <openssl/err.h>
#include <openssl/objects.h>
#include <openssl/ossl_typ.h>
#include <openssl/pem.h>
#include <openssl/rand.h>
#include <openssl/rsa.h>
#include <openssl/safestack.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

using namespace Kca::OpenSSL;

// Instantiate oneself to run startup code
const Utils *Utils::instance = new Utils();

// Filename of PRNG seed file
#ifdef __unix__
const char* Utils::random_source = "/dev/urandom";
#else
const char* Utils::random_source = ".rnd";
#endif

const ExtensionList Utils::emailCertConstraints(
    ExtensionList() <<
    Extension("subjectKeyIdentifier", "hash") <<
    Extension("authorityKeyIdentifier", "keyid,issuer:always") <<
    Extension("basicConstraints", "CA:false") <<
    Extension("keyUsage", "dataEncipherment, digitalSignature, keyEncipherment") <<
    Extension("extendedKeyUsage", "emailProtection") <<
    Extension("nsCertType", "email")
  );

Utils::Utils()
{
  // Load OpenSSL's library error strings.
  ERR_load_crypto_strings();
}

Utils::~Utils()
{
  // Unload error strings.
  ERR_free_strings();
}

QString Kca::OpenSSL::version()
{
  return QString(src_version);
}

QString Kca::OpenSSL::build_information()
{
  return QObject::tr("Commit %1 by %2, committed on %3.").arg(commit).arg(committer).arg(committer_date);
}

quint64 Kca::OpenSSL::random()
{
  union {
    quint64 number;
    unsigned char buffer[8];
  };

  if (!RAND_load_file(Utils::random_source, 64)) {
    return (quint64) ~0;
  }

  /* Doesn't need to be cryptographically strong. Just be random. */
  if (RAND_pseudo_bytes(buffer, 8) < 0) {
    return (quint64) ~0;
  }

  RAND_write_file(Utils::random_source);

  return number;
}

const EVP_MD* Utils::digest(const Digest digest)
{
  switch (digest) {
    case Kca::OpenSSL::RIPEMD160:
      return EVP_ripemd160();
      break;
    case Kca::OpenSSL::SHA1:
      return EVP_sha1();
      break;
    case Kca::OpenSSL::SHA256:
      return EVP_sha256();
      break;
    case Kca::OpenSSL::SHA384:
      return EVP_sha384();
      break;
    case Kca::OpenSSL::SHA512:
      return EVP_sha512();
      break;
  }

  return NULL;
}

ASN1_OCTET_STRING* Utils::octetString(const QString& in)
{
  ASN1_OCTET_STRING *result = ASN1_OCTET_STRING_new();
  if (!result)
    return NULL;

  QByteArray data = in.toLocal8Bit();
  if (!ASN1_OCTET_STRING_set(result,(const unsigned char *) data.constData(), data.length())) {
    ASN1_OCTET_STRING_free(result);
    return NULL;
  }

  return result;
}

X509_NAME* Utils::x509name(const QByteArray& name)
{
  X509_NAME *result = X509_NAME_new();
  if (!result)
    return NULL;

  int key;
  QStringList pairs, values;

  QStringList entries = QString(name).split('/', QString::SkipEmptyParts);
  for(QList< QString >::const_iterator it = entries.constBegin();
      it != entries.constEnd(); ++it) {
    pairs = (*it).split('=', QString::SkipEmptyParts);
    if (pairs.length() < 2)
      continue;

    key = OBJ_txt2nid(pairs[0].toLocal8Bit().constData());
    values = pairs.mid(1);

    for (QList< QString >::const_iterator entry = values.constBegin();
         entry != values.constEnd(); ++entry) {
      X509_NAME_add_entry_by_NID(result, key, MBSTRING_UTF8,
          (unsigned char *) (*entry).toUtf8().constData(), (*entry).length(), -1, 0);
    }
  }

  return result;
}

ExtensionList Kca::OpenSSL::emailCertExtensions()
{
  return Utils::emailCertConstraints;
}

ExtensionList Utils::extensions(STACK_OF(X509_EXTENSION)* extensions)
{
  if (!extensions)
    return ExtensionList();

  ExtensionList result = ExtensionList();
  X509_EXTENSION *ext;

  for (int i=0; i < extensions->stack.num; ++i) {
    ext = sk_X509_EXTENSION_value(extensions, i);
    if (!ext)
      continue;

    result << Extension(ext);
  }

  return result;
}
