/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#define _KCAOSSL_UNIT_TESTABLE friend class RequestTest;

#define KEY_LENGTH 3072

#include "../request.h"

#include "../extension.h"
#include "../key.h"
#include "../opensslexception.h"

#include <QtCore/QByteArray>
#include <QtCore/QObject>
#include <QtCore/QDebug>

#include <QtTest/QTest>

#include <openssl/x509.h>

static const char *pem = "-----BEGIN CERTIFICATE REQUEST-----\n"
"MIIB8jCCAVsCAQAwfjELMAkGA1UEBhMCREUxEDAOBgNVBAgTB0hhbWJ1cmcxEDAO\n"
"BgNVBAcTB0hhbWJ1cmcxETAPBgNVBAoTCFBDIFRpZWRlMR0wGwYDVQQLExRTb2Z0\n"
"d2FyZSBEZXZlbG9wbWVudDEZMBcGA1UEAxQQbGlia2NhX29zc2wgVGVzdDCBnzAN\n"
"BgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAwgrtYXbm1GBDOJQmvRKpgnvBVdMPpcOw\n"
"0qns9OhJSAjL3PQZxQ0MwMA/+fpz1KDeXAHu5dWZVC5Wzm4WKB9KVH+9UMWC1x30\n"
"Rx3laMhIDQZSs7cNDWo/K2jte+T+kGhz3FWdcx1Feq0FMWoPImJy5z6tftTxEHRN\n"
"dBhB0csRivUCAwEAAaA0MDIGCSqGSIb3DQEJDjElMCMwIQYLKwYBBAGB4zAUFAEE\n"
"EgwQVW5pdCB0ZXN0IHN0cmluZzANBgkqhkiG9w0BAQUFAAOBgQAbqECXemMkaZNW\n"
"WBp3z1JBDSWG6ahk0cNWyERcqSlZ8WaeDjTMSK9icRiyxoJn2pTRbcne9kG1KSKt\n"
"0QZcUUrE2ng6NHSUZ3djqp/VbKyFhJhF5CXuSPgtp9zii3xDCfp64IidUyonvkuw\n"
"pRUb7nLsbDX1EMMUN1Vhmt13FEuSdA==\n"
"-----END CERTIFICATE REQUEST-----\n";

static const char der[502] = {
  '\x30','\x82','\x01','\xf2','\x30','\x82','\x01','\x5b','\x02','\x01','\x00','\x30',
  '\x7e','\x31','\x0b','\x30','\x09','\x06','\x03','\x55','\x04','\x06','\x13','\x02',
  '\x44','\x45','\x31','\x10','\x30','\x0e','\x06','\x03','\x55','\x04','\x08','\x13',
  '\x07','\x48','\x61','\x6d','\x62','\x75','\x72','\x67','\x31','\x10','\x30','\x0e',
  '\x06','\x03','\x55','\x04','\x07','\x13','\x07','\x48','\x61','\x6d','\x62','\x75',
  '\x72','\x67','\x31','\x11','\x30','\x0f','\x06','\x03','\x55','\x04','\x0a','\x13',
  '\x08','\x50','\x43','\x20','\x54','\x69','\x65','\x64','\x65','\x31','\x1d','\x30',
  '\x1b','\x06','\x03','\x55','\x04','\x0b','\x13','\x14','\x53','\x6f','\x66','\x74',
  '\x77','\x61','\x72','\x65','\x20','\x44','\x65','\x76','\x65','\x6c','\x6f','\x70',
  '\x6d','\x65','\x6e','\x74','\x31','\x19','\x30','\x17','\x06','\x03','\x55','\x04',
  '\x03','\x14','\x10','\x6c','\x69','\x62','\x6b','\x63','\x61','\x5f','\x6f','\x73',
  '\x73','\x6c','\x20','\x54','\x65','\x73','\x74','\x30','\x81','\x9f','\x30','\x0d',
  '\x06','\x09','\x2a','\x86','\x48','\x86','\xf7','\x0d','\x01','\x01','\x01','\x05',
  '\x00','\x03','\x81','\x8d','\x00','\x30','\x81','\x89','\x02','\x81','\x81','\x00',
  '\xc2','\x0a','\xed','\x61','\x76','\xe6','\xd4','\x60','\x43','\x38','\x94','\x26',
  '\xbd','\x12','\xa9','\x82','\x7b','\xc1','\x55','\xd3','\x0f','\xa5','\xc3','\xb0',
  '\xd2','\xa9','\xec','\xf4','\xe8','\x49','\x48','\x08','\xcb','\xdc','\xf4','\x19',
  '\xc5','\x0d','\x0c','\xc0','\xc0','\x3f','\xf9','\xfa','\x73','\xd4','\xa0','\xde',
  '\x5c','\x01','\xee','\xe5','\xd5','\x99','\x54','\x2e','\x56','\xce','\x6e','\x16',
  '\x28','\x1f','\x4a','\x54','\x7f','\xbd','\x50','\xc5','\x82','\xd7','\x1d','\xf4',
  '\x47','\x1d','\xe5','\x68','\xc8','\x48','\x0d','\x06','\x52','\xb3','\xb7','\x0d',
  '\x0d','\x6a','\x3f','\x2b','\x68','\xed','\x7b','\xe4','\xfe','\x90','\x68','\x73',
  '\xdc','\x55','\x9d','\x73','\x1d','\x45','\x7a','\xad','\x05','\x31','\x6a','\x0f',
  '\x22','\x62','\x72','\xe7','\x3e','\xad','\x7e','\xd4','\xf1','\x10','\x74','\x4d',
  '\x74','\x18','\x41','\xd1','\xcb','\x11','\x8a','\xf5','\x02','\x03','\x01','\x00',
  '\x01','\xa0','\x34','\x30','\x32','\x06','\x09','\x2a','\x86','\x48','\x86','\xf7',
  '\x0d','\x01','\x09','\x0e','\x31','\x25','\x30','\x23','\x30','\x21','\x06','\x0b',
  '\x2b','\x06','\x01','\x04','\x01','\x81','\xe3','\x30','\x14','\x14','\x01','\x04',
  '\x12','\x0c','\x10','\x55','\x6e','\x69','\x74','\x20','\x74','\x65','\x73','\x74',
  '\x20','\x73','\x74','\x72','\x69','\x6e','\x67','\x30','\x0d','\x06','\x09','\x2a',
  '\x86','\x48','\x86','\xf7','\x0d','\x01','\x01','\x05','\x05','\x00','\x03','\x81',
  '\x81','\x00','\x1b','\xa8','\x40','\x97','\x7a','\x63','\x24','\x69','\x93','\x56',
  '\x58','\x1a','\x77','\xcf','\x52','\x41','\x0d','\x25','\x86','\xe9','\xa8','\x64',
  '\xd1','\xc3','\x56','\xc8','\x44','\x5c','\xa9','\x29','\x59','\xf1','\x66','\x9e',
  '\x0e','\x34','\xcc','\x48','\xaf','\x62','\x71','\x18','\xb2','\xc6','\x82','\x67',
  '\xda','\x94','\xd1','\x6d','\xc9','\xde','\xf6','\x41','\xb5','\x29','\x22','\xad',
  '\xd1','\x06','\x5c','\x51','\x4a','\xc4','\xda','\x78','\x3a','\x34','\x74','\x94',
  '\x67','\x77','\x63','\xaa','\x9f','\xd5','\x6c','\xac','\x85','\x84','\x98','\x45',
  '\xe4','\x25','\xee','\x48','\xf8','\x2d','\xa7','\xdc','\xe2','\x8b','\x7c','\x43',
  '\x09','\xfa','\x7a','\xe0','\x88','\x9d','\x53','\x2a','\x27','\xbe','\x4b','\xb0',
  '\xa5','\x15','\x1b','\xee','\x72','\xec','\x6c','\x35','\xf5','\x10','\xc3','\x14',
  '\x37','\x55','\x61','\x9a','\xdd','\x77','\x14','\x4b','\x92','\x74',
};

namespace Kca {
namespace OpenSSL {

class RequestTest : public QObject
{
  Q_OBJECT

private slots:
  void initTestCase()
  {
    subject = "/C=DE/ST=Hamburg/L=Hamburg/O=PC Tiede/OU=Software Development/CN=libkca_ossl Test";
    testKey = Key::generateKeyPair();
  }

  void testStaticPEMRequest()
  {
    Request pemRequest = Request(QByteArray(pem));
    ExtensionList list = pemRequest.extensions();

    QVERIFY(!pemRequest.isNull());
    QCOMPARE(pemRequest.version(), QByteArray("0"));
    QCOMPARE(pemRequest.subject(), this->subject);

    QCOMPARE(pemRequest.toPem(), QByteArray(pem));
    X509_REQ *handle = pemRequest.handle();
    QVERIFY(handle);
    if (handle)
      X509_REQ_free(handle);

    QCOMPARE(list.length(), 1);
    QCOMPARE(list[0].oid().oid, QString("1.3.6.1.4.1.29104.20.20.1"));
    QCOMPARE(list[0].oid().shortName, QString("UNDEF"));
    QCOMPARE(list[0].oid().longName, QString("undefined"));
    QCOMPARE(list[0].critical(), false);
    QCOMPARE(list[0].value(), QString("..Unit test string"));
  }

  void testStaticDERRequest()
  {
    Request derRequest = Request(QByteArray(der, 502), QSsl::Der);
    ExtensionList list = derRequest.extensions();

    QVERIFY(!derRequest.isNull());
    QCOMPARE(derRequest.version(), QByteArray("0"));
    QCOMPARE(derRequest.subject(), this->subject);

    QCOMPARE(derRequest.toDer(), QByteArray(der, 502));
    X509_REQ *handle = derRequest.handle();
    QVERIFY(handle);
    if (handle)
      X509_REQ_free(handle);

    QCOMPARE(list.length(), 1);
    QCOMPARE(list[0].oid().oid, QString("1.3.6.1.4.1.29104.20.20.1"));
    QCOMPARE(list[0].oid().shortName, QString("UNDEF"));
    QCOMPARE(list[0].oid().longName, QString("undefined"));
    QCOMPARE(list[0].critical(), false);
    QCOMPARE(list[0].value(), QString("..Unit test string"));
  }

  void testComparison()
  {
    Request pemRequest = Request(QByteArray(pem));
    Request derRequest = Request(QByteArray(der, 502), QSsl::Der);

    QVERIFY(pemRequest == derRequest);
  }

  void testGenerateRequest()
  {
    Request generated = Request::generate(testKey, subject);

    QVERIFY(!generated.isNull());
    QCOMPARE(generated.subject(), subject);

    QCOMPARE(generated.extensions().length(), 0);
  }

  void testGenerateRequestWithExtensions()
  {
    Request request;

    ExtensionList ext = ExtensionList();
    ext << Extension("subjectKeyIdentifier", "hash");
    ext << Extension("subjectAltName", "DNS:test.example.com");

    try {
      request = Request::generate(testKey, subject, ext);
    }
    catch (OpenSSLException& ex) {
      QFAIL(QString("Got OpenSSLException: %1 @ %2").arg(ex.what()).arg(ex.where()).toUtf8().constData());
    }
    QVERIFY(!request.isNull());

    QCOMPARE(request.subject(), subject);
    QCOMPARE(request.version(), QByteArray("2"));
    QVERIFY(request.extensions().length() != 0);

    ext << Extension("authorityKeyIdentifier", "keyid:always,issuer:always");
    try {
      request = Request::generate(testKey, subject, ext);
      QFAIL("Expected exception not thrown.");
    }
    catch (OpenSSLException& ex) {
      QVERIFY(strlen(ex.what()));
      qDebug() << QString("%1 @ %2").arg(ex.what()).arg(ex.where());
    }
  }

  void cleanupTestCase()
  {
    testKey.clear();
  }

private:
  QByteArray subject;

  Key testKey;

};  /* End class RequestTest */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

QTEST_MAIN(Kca::OpenSSL::RequestTest);
#include "requesttest.moc"
