/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#define _KCAOSSL_UNIT_TESTABLE friend class KeyTest;

#include "../key.h"

#include "../opensslexception.h"

#include <QtNetwork/QSslKey>

#include <openssl/evp.h>

#include <QtTest/QTest>

static const char *pem = "-----BEGIN PRIVATE KEY-----\n"
"MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALIWSApz2x3XeFIM\n"
"s0XmKQrnp4ugAHzyITwYH2DxkGBeYo+4VZZG9soeKJ98kIyfIGL/+xU/JXO/VfLI\n"
"x19ICZ3xaWr3ZFOkqdvBqiKlT1ARcsIGR9/IgPNd0aTAKu08QGrjhecLX+INhZTh\n"
"gsWBvqW5NKWCR04CtW3h/JuNpgEXAgMBAAECgYA22joSTiEdjfKXgyXsz7mwjC3F\n"
"CeFcms+1zNBNcdpAdJHzJpnmEbGOC8ADb8CuL40wMRxLLQoCHtsATutp46xiH9aa\n"
"TtIi6tGUrmgDnHF/7oFMPitEKyDqM1MbR/iZ2HdCFK350aIVffisEtLtxfDkmfH2\n"
"spiWIm4JkZSeQs2nEQJBAORktEnIx+OsSx/0FRusSCmwSvFCp3feIiysULACnIZL\n"
"NuXj/JkFmNZ9G4+69CLCVXrO60eWCCBvGIGB5Yi+UIUCQQDHnOv5uDxtnMGpIk6D\n"
"w34AJPHM6pvWuOr1eQwxYs6nymgqX8KHTf7YwKrjJ6lI4o/fJeC5kqY0MtqYEzo5\n"
"aOvrAkATJJGr4YVHguhlDDs8r4r424rcf96zgE3S+rpklkNyShmW2S4R0VNHHwNR\n"
"X7UYuLsbaP51aJDTzxv5tt57QKBVAkA5CrbEXcmm6SzOUGdvb+YJ1ITksgZmjeFW\n"
"1IiPdFTpzgWGVmcYk0lIV4Z980+E7HmAoypc3XNxZzbgTl5NGZaRAkEAy2c0xnj/\n"
"emOGAYU5TYayu3P/M8EHWFkquBbr4ML5l6IE1wK27KMwM8rCfFgmfD4IRtRaTgMO\n"
"AIS8Sh10fRGgYw==\n"
"-----END PRIVATE KEY-----\n";

static const char der[608] = {
  '\x30','\x82','\x02','\x5c','\x02','\x01','\x00','\x02','\x81','\x81','\x00','\xb2',
  '\x16','\x48','\x0a','\x73','\xdb','\x1d','\xd7','\x78','\x52','\x0c','\xb3','\x45',
  '\xe6','\x29','\x0a','\xe7','\xa7','\x8b','\xa0','\x00','\x7c','\xf2','\x21','\x3c',
  '\x18','\x1f','\x60','\xf1','\x90','\x60','\x5e','\x62','\x8f','\xb8','\x55','\x96',
  '\x46','\xf6','\xca','\x1e','\x28','\x9f','\x7c','\x90','\x8c','\x9f','\x20','\x62',
  '\xff','\xfb','\x15','\x3f','\x25','\x73','\xbf','\x55','\xf2','\xc8','\xc7','\x5f',
  '\x48','\x09','\x9d','\xf1','\x69','\x6a','\xf7','\x64','\x53','\xa4','\xa9','\xdb',
  '\xc1','\xaa','\x22','\xa5','\x4f','\x50','\x11','\x72','\xc2','\x06','\x47','\xdf',
  '\xc8','\x80','\xf3','\x5d','\xd1','\xa4','\xc0','\x2a','\xed','\x3c','\x40','\x6a',
  '\xe3','\x85','\xe7','\x0b','\x5f','\xe2','\x0d','\x85','\x94','\xe1','\x82','\xc5',
  '\x81','\xbe','\xa5','\xb9','\x34','\xa5','\x82','\x47','\x4e','\x02','\xb5','\x6d',
  '\xe1','\xfc','\x9b','\x8d','\xa6','\x01','\x17','\x02','\x03','\x01','\x00','\x01',
  '\x02','\x81','\x80','\x36','\xda','\x3a','\x12','\x4e','\x21','\x1d','\x8d','\xf2',
  '\x97','\x83','\x25','\xec','\xcf','\xb9','\xb0','\x8c','\x2d','\xc5','\x09','\xe1',
  '\x5c','\x9a','\xcf','\xb5','\xcc','\xd0','\x4d','\x71','\xda','\x40','\x74','\x91',
  '\xf3','\x26','\x99','\xe6','\x11','\xb1','\x8e','\x0b','\xc0','\x03','\x6f','\xc0',
  '\xae','\x2f','\x8d','\x30','\x31','\x1c','\x4b','\x2d','\x0a','\x02','\x1e','\xdb',
  '\x00','\x4e','\xeb','\x69','\xe3','\xac','\x62','\x1f','\xd6','\x9a','\x4e','\xd2',
  '\x22','\xea','\xd1','\x94','\xae','\x68','\x03','\x9c','\x71','\x7f','\xee','\x81',
  '\x4c','\x3e','\x2b','\x44','\x2b','\x20','\xea','\x33','\x53','\x1b','\x47','\xf8',
  '\x99','\xd8','\x77','\x42','\x14','\xad','\xf9','\xd1','\xa2','\x15','\x7d','\xf8',
  '\xac','\x12','\xd2','\xed','\xc5','\xf0','\xe4','\x99','\xf1','\xf6','\xb2','\x98',
  '\x96','\x22','\x6e','\x09','\x91','\x94','\x9e','\x42','\xcd','\xa7','\x11','\x02',
  '\x41','\x00','\xe4','\x64','\xb4','\x49','\xc8','\xc7','\xe3','\xac','\x4b','\x1f',
  '\xf4','\x15','\x1b','\xac','\x48','\x29','\xb0','\x4a','\xf1','\x42','\xa7','\x77',
  '\xde','\x22','\x2c','\xac','\x50','\xb0','\x02','\x9c','\x86','\x4b','\x36','\xe5',
  '\xe3','\xfc','\x99','\x05','\x98','\xd6','\x7d','\x1b','\x8f','\xba','\xf4','\x22',
  '\xc2','\x55','\x7a','\xce','\xeb','\x47','\x96','\x08','\x20','\x6f','\x18','\x81',
  '\x81','\xe5','\x88','\xbe','\x50','\x85','\x02','\x41','\x00','\xc7','\x9c','\xeb',
  '\xf9','\xb8','\x3c','\x6d','\x9c','\xc1','\xa9','\x22','\x4e','\x83','\xc3','\x7e',
  '\x00','\x24','\xf1','\xcc','\xea','\x9b','\xd6','\xb8','\xea','\xf5','\x79','\x0c',
  '\x31','\x62','\xce','\xa7','\xca','\x68','\x2a','\x5f','\xc2','\x87','\x4d','\xfe',
  '\xd8','\xc0','\xaa','\xe3','\x27','\xa9','\x48','\xe2','\x8f','\xdf','\x25','\xe0',
  '\xb9','\x92','\xa6','\x34','\x32','\xda','\x98','\x13','\x3a','\x39','\x68','\xeb',
  '\xeb','\x02','\x40','\x13','\x24','\x91','\xab','\xe1','\x85','\x47','\x82','\xe8',
  '\x65','\x0c','\x3b','\x3c','\xaf','\x8a','\xf8','\xdb','\x8a','\xdc','\x7f','\xde',
  '\xb3','\x80','\x4d','\xd2','\xfa','\xba','\x64','\x96','\x43','\x72','\x4a','\x19',
  '\x96','\xd9','\x2e','\x11','\xd1','\x53','\x47','\x1f','\x03','\x51','\x5f','\xb5',
  '\x18','\xb8','\xbb','\x1b','\x68','\xfe','\x75','\x68','\x90','\xd3','\xcf','\x1b',
  '\xf9','\xb6','\xde','\x7b','\x40','\xa0','\x55','\x02','\x40','\x39','\x0a','\xb6',
  '\xc4','\x5d','\xc9','\xa6','\xe9','\x2c','\xce','\x50','\x67','\x6f','\x6f','\xe6',
  '\x09','\xd4','\x84','\xe4','\xb2','\x06','\x66','\x8d','\xe1','\x56','\xd4','\x88',
  '\x8f','\x74','\x54','\xe9','\xce','\x05','\x86','\x56','\x67','\x18','\x93','\x49',
  '\x48','\x57','\x86','\x7d','\xf3','\x4f','\x84','\xec','\x79','\x80','\xa3','\x2a',
  '\x5c','\xdd','\x73','\x71','\x67','\x36','\xe0','\x4e','\x5e','\x4d','\x19','\x96',
  '\x91','\x02','\x41','\x00','\xcb','\x67','\x34','\xc6','\x78','\xff','\x7a','\x63',
  '\x86','\x01','\x85','\x39','\x4d','\x86','\xb2','\xbb','\x73','\xff','\x33','\xc1',
  '\x07','\x58','\x59','\x2a','\xb8','\x16','\xeb','\xe0','\xc2','\xf9','\x97','\xa2',
  '\x04','\xd7','\x02','\xb6','\xec','\xa3','\x30','\x33','\xca','\xc2','\x7c','\x58',
  '\x26','\x7c','\x3e','\x08','\x46','\xd4','\x5a','\x4e','\x03','\x0e','\x00','\x84',
  '\xbc','\x4a','\x1d','\x74','\x7d','\x11','\xa0','\x63',
};

namespace Kca {
namespace OpenSSL {

class KeyTest : public QObject
{
  Q_OBJECT

private slots:
  void testConstructor()
  {
    Key key = Key();
    QVERIFY(key.isNull());

    Key pemEncoded(QSslKey(QByteArray(pem), QSsl::Rsa, QSsl::Pem, QSsl::PrivateKey));
    QVERIFY(!pemEncoded.isNull());

    Key derEncoded(QSslKey(QByteArray(der, 608), QSsl::Rsa, QSsl::Der, QSsl::PrivateKey));
    QVERIFY(!derEncoded.isNull());

    QCOMPARE(pemEncoded, derEncoded);
  }

  void testCopyConstructor()
  {
    Key src(QSslKey(QByteArray(der, 608), QSsl::Rsa, QSsl::Der, QSsl::PrivateKey));
    Key dst(src);

    QVERIFY(!src.isNull());
    QVERIFY(!dst.isNull());
    QCOMPARE(src, dst);
  }

  void testAssignment()
  {
    Key src(QSslKey(QByteArray(der, 608), QSsl::Rsa, QSsl::Der, QSsl::PrivateKey));
    Key dst;

    QVERIFY(!src.isNull());
    QVERIFY(dst.isNull());
    QVERIFY(src != dst);

    dst = src;
    QVERIFY(!dst.isNull());
    QVERIFY(src == dst);
  }

  void testGenerate()
  {
    // DSA key generation is not supported, make sure it _does_ fail!
    try {
      Key dsa = Key::generateKeyPair(768, QSsl::Dsa);
      QVERIFY(dsa.isNull());

      Key key = Key::generateKeyPair(1024, QSsl::Rsa);
      QVERIFY(!key.isNull());
      QCOMPARE(key.length(), 1024);
      QCOMPARE(key.algorithm(), QSsl::Rsa);
      QCOMPARE(key.type(), QSsl::PrivateKey);
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testHandle()
  {
    try {
      Key key = Key(QSslKey(QByteArray(der, 608), QSsl::Rsa, QSsl::Der));

      EVP_PKEY *pkey = key.handle();
      QVERIFY(pkey);
      EVP_PKEY_free(pkey);
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
    try {
      // Use default values, should be 2048 for length and RSA as algorithm
      Key key = Key::generateKeyPair();
      QVERIFY(!key.isNull());
      QCOMPARE(key.length(), 2048);
      QCOMPARE(key.algorithm(), QSsl::Rsa);
      QCOMPARE(key.type(), QSsl::PrivateKey);

      EVP_PKEY *pkey = key.handle();
      QVERIFY(pkey);
      EVP_PKEY_free(pkey);
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }
};  /* End class KeyTest */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

QTEST_MAIN(Kca::OpenSSL::KeyTest);
#include "keytest.moc"
