/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#define _KCAOSSL_UNIT_TESTABLE friend class ExtensionTest;

#include "../extension.h"
#include "../opensslexception.h"

#include <QtCore/QObject>
#include <QtCore/QDebug>

#include <QtTest/QTest>

#include <openssl/objects.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

namespace Kca {
namespace OpenSSL {

/**
 * Unit test for custom exception class Extension.
 *
 * @author Felix Tiede <info@pc-tiede.de>
 * @short Extension unit test.
 */
class ExtensionTest : public QObject {
  Q_OBJECT

private slots:
  /** Test constructor by checking return values match input. */
  void testConstructor()
  {
    Extension::ObjectID oid;
    oid.shortName = "basicConstraints";
    oid.longName = "X509v3 Basic Constraints";

    Extension ext(oid, "CA:false", true);

    QCOMPARE(ext.critical(), true);
    QCOMPARE(ext.value(), QString("CA:false"));
    QCOMPARE(ext.oid().shortName, oid.shortName);
  };
  /** Test comparison operation. */
  void testComparison()
  {
    Extension::ObjectID oid;
    oid.shortName = "basicConstraints";
    oid.longName = "X509v3 Basic Constraints";

    Extension src(oid, "CA:false", true);
    Extension dst(oid, "CA:false", true);
    Extension mis(oid, "CA:true", true);

    QVERIFY(!src.value().isEmpty());
    QVERIFY(!dst.value().isEmpty());
    QVERIFY(!mis.value().isEmpty());

    QVERIFY(src == dst);
    QVERIFY(src != mis);
    QVERIFY(dst != mis);
  }
  /** Test OID registration in constructor. */
  void testOIDRegistration()
  {
    Extension::ObjectID oid;
    oid.oid       = "1.3.6.1.4.1.29104.20.20.1";
    oid.shortName = "softwareTesting";
    oid.longName  = "Software unit test extension";
    try {
      Extension ext(oid, "some input string", false);
      QCOMPARE(ext.critical(), false);
      QCOMPARE(ext.value(), QString("some input string"));
      QCOMPARE(ext.oid().oid, oid.oid);

      X509_EXTENSION *handle = ext.handle();
      if (handle) {
        QVERIFY(handle->object->nid != NID_undef);
        registeredNid = handle->object->nid;
        X509_EXTENSION_free(handle);
      }
    }
    catch (OpenSSLException& e) {
      QFAIL(e.what());
    }
  };
  /** Test protected name lookup constructor with and without exception. */
  void testNameLookupConstructor()
  {
    try {
      Extension ext("basicConstraints", "CA:false", true);
      QCOMPARE(ext.critical(), true);
      QCOMPARE(ext.value(), QString("CA:false"));
      QCOMPARE(ext.oid().shortName, QString("basicConstraints"));
      QCOMPARE(ext.oid().longName, QString("X509v3 Basic Constraints"));
    }
    catch (OpenSSLException& e) {
      QFAIL(e.what());
    }

    try {
      Extension registered("softwareTesting", "some string", false);
      QCOMPARE(registered.critical(), false);
      QCOMPARE(registered.value(), QString("some string"));
      QCOMPARE(registered.oid().oid, QString("1.3.6.1.4.1.29104.20.20.1"));
      QCOMPARE(registered.oid().shortName, QString("softwareTesting"));
      QCOMPARE(registered.oid().longName, QString("Software unit test extension"));
    }
    catch (OpenSSLException& e) {
      QFAIL(e.what());
    }

    try {
      Extension failure("Something_Unregistered", "foobar", true);
      // These should never be called as the constructor should throw an exception.
      QFAIL("Expected exception not thrown.");
    }
    catch (OpenSSLException& e) {
      QVERIFY(strlen(e.what()));
      qDebug() << e.what() << "@" << e.where();
    }
  };
  /** Test protected NID lookup constructor with and without exception. */
  void testNIDLookupConstructor()
  {
    try {
      Extension ext(NID_basic_constraints, "CA:false", true);
      QCOMPARE(ext.critical(), true);
      QCOMPARE(ext.value(), QString("CA:false"));
      QCOMPARE(ext.oid().shortName, QString("basicConstraints"));
      QCOMPARE(ext.oid().longName, QString("X509v3 Basic Constraints"));
    }
    catch (OpenSSLException& e) {
      QFAIL(e.what());
    }

    try {
      Extension ext(registeredNid, "some other string", true);
      QCOMPARE(ext.critical(), true);
      QCOMPARE(ext.value(), QString("some other string"));
      QCOMPARE(ext.oid().oid, QString("1.3.6.1.4.1.29104.20.20.1"));
      QCOMPARE(ext.oid().shortName, QString("softwareTesting"));
      QCOMPARE(ext.oid().longName, QString("Software unit test extension"));
    }
    catch (OpenSSLException& e) {
      QFAIL(e.what());
    }

    try {
      Extension failure(NID_undef, "foobar", true);
      // These should never be called as the constructor should throw an exception.
      QFAIL("Expected exception not thrown.");
    }
    catch (OpenSSLException& e) {
      QVERIFY(strlen(e.what()));
      qDebug() << e.what() << "@" << e.where();
    }
  };

  /** Test failure setter method. */
  void testValueSetter()
  {
    Extension ext("basicConstraints", "CA:false");

    QCOMPARE(ext.value(), QString("CA:false"));

    ext.setValue("CA:true");
    QCOMPARE(ext.value(), QString("CA:true"));
  };

  /** Test description setter method. */
  void testCriticalitySetter()
  {
    Extension ext("basicConstraints", "CA:false");

    QCOMPARE(ext.critical(), false);

    ext.setCritical(true);
    QCOMPARE(ext.critical(), true);
  };

  /** Test setter for replace flag. */
  void testReplaceSetter()
  {
    Extension ext("basicConstraints", "CA:false");

    QCOMPARE(ext.replace(), false);

    ext.setReplace(true);
    QCOMPARE(ext.replace(), true);
  };

  /** Test getter for OpenSSL handle. */
  void testHandleGetter()
  {
    try {
      X509_EXTENSION *handle;
      Extension ext("basicConstraints", "CA:false");

      handle = ext.handle();
      QVERIFY(handle != NULL);
      if (handle)
        X509_EXTENSION_free(handle);

      X509V3_CTX v3_ctx;
      X509V3_set_ctx_nodb(&v3_ctx);
      X509V3_set_ctx(&v3_ctx, NULL, NULL, NULL, NULL, 0);
      handle = ext.handle(&v3_ctx);
      QVERIFY(handle != NULL);
      if (handle)
        X509_EXTENSION_free(handle);
    }
    catch (OpenSSLException& e) {
      QVERIFY(strlen(e.what()));
      qDebug() << e.what() << "@" << e.where();
    }
  };

  /** Test constructor from OpenSSL handle. */
  void testHandleConstructor()
  {
    try {
      X509_EXTENSION *handle;
      Extension ext("basicConstraints", "CA:false");

      handle = ext.handle();
      QVERIFY(handle != NULL);

      Extension fromHandle(handle);
      QCOMPARE(fromHandle.oid().shortName, QString("basicConstraints"));
      QCOMPARE(fromHandle.critical(), false);
      QCOMPARE(fromHandle.value(), QString("CA:false"));

      X509_EXTENSION_free(handle);
    }
    catch (OpenSSLException& e) {
      QFAIL(e.what());
    }
  };

private:
  int registeredNid;
};  /* End class SigningExceptionTest */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

QTEST_MAIN(Kca::OpenSSL::ExtensionTest)
#include "extensiontest.moc"
