/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#define _KCAOSSL_UNIT_TESTABLE friend class CertificateTest;

#include "../certificate.h"

#include "../common.h"
#include "../extension.h"
#include "../key.h"
#include "../opensslexception.h"
#include "../signingexception.h"
#include "../request.h"

#include <QtCore/QByteArray>
#include <QtCore/QDateTime>
#include <QtCore/QObject>
#include <QtCore/QString>

#include <QtNetwork/QSslKey>

#include <QtTest/QTest>

#include <QDebug>

namespace Kca {
namespace OpenSSL {

class CertificateTest : public QObject
{
  Q_OBJECT

private slots:
  void initTestCase()
  {
    subject = "/C=DE/ST=Hamburg/O=Felix Tiede/OU=Software Development/CN=Unit test certificate/";

    defaults.serial = random();
    defaults.digest = SHA256;
    defaults.effectiveDate = QDateTime::currentDateTime();
    defaults.expiryDate    = defaults.effectiveDate.addSecs(45);
    crl = CRL()
          << CRLEntry { random(), CessationOfOperation, QDateTime::currentDateTime().addDays(-2) }
          << CRLEntry { random(), CACompromise, QDateTime::currentDateTime().addDays(-1) };

    Certificate::SignatureDetails details;
    details.serial = random();
    details.digest = SHA256;
    details.effectiveDate = QDateTime::currentDateTime();
    details.expiryDate    = details.effectiveDate.addSecs(60);

    ExtensionList extensions;
    extensions << Extension("basicConstraints", "CA:true", true)
               << Extension("keyUsage", "keyCertSign, cRLSign")
               << Extension("subjectKeyIdentifier", "hash");

    customOID.oid       = "1.3.6.1.4.1.29104.20.20.1";
    customOID.shortName = "softwareTesting";
    customOID.longName  = "Software unit test extension";
    extensions << Extension(customOID, "Test value", false);

    try {
      caKey = Key::generateKeyPair();
      caCert = Certificate(caKey, subject, details, extensions);

      requestKey = Key::generateKeyPair();
      request = Request::generate(requestKey,
                                  "/C=DE/ST=Hamburg/O=Felix Tiede/OU=Software Development/CN=Unit test/",
                                  ExtensionList() << Extension("subjectAltName",
                                                               "URI:http://test.pc-tiede.de/,IP:127.0.0.1"),
                                  SHA512);

      details.serial = random();
      stdKey = Key::generateKeyPair();
      stdCert = Certificate(stdKey, subject, details, emailCertExtensions());
    }
    catch (SigningException& e) {
      QFAIL(QString("Signing exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testGlobals()
  {
    QVERIFY(!caKey.isNull());

    QVERIFY(!caCert.isNull());
    QVERIFY(caCert.isValid());
    QVERIFY(caCert.isCA());

    QVERIFY(!requestKey.isNull());
    QVERIFY(!request.isNull());

    QVERIFY(!stdCert.isCA());

    QCOMPARE(crl.count(), 2);
  }

  void testKeyMatch()
  {
    QVERIFY(caCert.keyMatch(caKey));
    QVERIFY(!stdCert.keyMatch(caKey));
    QVERIFY(stdCert.keyMatch(stdKey));
  }

  void testSelfSignKeyRestraint()
  {
    try {
      // Empty signature defaults, it shouldn't even get to examine them.
      Certificate test(QSslKey(), subject, Certificate::SignatureDetails(), emailCertExtensions());
      QFAIL("Expected exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::KeyMismatch);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testSelfSignTimeRestraints()
  {
    QSslKey key = Key::generateKeyPair();

    Certificate::SignatureDetails details(defaults);
    details.serial = random();
    details.expiryDate    = details.effectiveDate.addSecs(-60);

    try {
      Certificate cert(key, subject, details, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }

    details.effectiveDate = details.expiryDate.addSecs(-60);
    try {
      Certificate cert(key, subject, details, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testSelfSignConstructor()
  {
    QSslKey key = Key::generateKeyPair();

    try {
      Certificate cert(key, subject, defaults, emailCertExtensions());
      QVERIFY(!cert.isNull());
      QVERIFY(cert.isValid());
    }
    catch (SigningException& e) {
      QFAIL(QString("Signing exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testExtensions()
  {
    ExtensionList caExtensions = caCert.extensions();
    QVERIFY(!caExtensions.isEmpty());

    ExtensionList stdExtensions = stdCert.extensions();
    QVERIFY(!stdExtensions.isEmpty());

    QVERIFY(caExtensions != stdExtensions);
  }

  void testSignCARestraint()
  {
    try {
      Certificate cert = stdCert.sign(request, stdKey, defaults, emailCertExtensions());
      QFAIL("Expected exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::NoCACertificate);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testSignTimeRestraints()
  {
    Certificate::SignatureDetails details(defaults);
    details.expiryDate = details.effectiveDate.addSecs(-30);
    try {
      Certificate cert = caCert.sign(request, caKey, details, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }

    details.effectiveDate = caCert.effectiveDate().addSecs(-1);
    details.expiryDate = defaults.expiryDate;
    try {
      Certificate cert = caCert.sign(request, caKey, details, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }

    details.effectiveDate = caCert.effectiveDate();
    details.expiryDate = caCert.expiryDate().addSecs(1);
    try {
      Certificate cert = caCert.sign(request, caKey, details, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testSignKeyRestraints()
  {
    try {
      Certificate cert = caCert.sign(request, QSslKey(), defaults, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::KeyMismatch);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }

    try {
      Certificate cert = caCert.sign(request, stdKey, defaults, emailCertExtensions());
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCsr);
      QCOMPARE(e.failure(), SigningException::KeyMismatch);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testRequestSigning()
  {
    try {
      Certificate cert = caCert.sign(request, caKey, defaults, emailCertExtensions());

      QVERIFY(!cert.isNull());
      QVERIFY(cert.isValid());

      QVERIFY(cert.effectiveDate().secsTo(defaults.effectiveDate) < 1);
      QVERIFY(cert.expiryDate().secsTo(defaults.expiryDate) < 1);

      Certificate certCopy(cert);
      QVERIFY(!certCopy.isCA());
      QVERIFY(certCopy.keyMatch(requestKey));
      QVERIFY(!certCopy.keyMatch(caKey));
    }
    catch(SigningException& e) {
      QFAIL(QString("Signing exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
    catch(OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testCRLSignCARestraint()
  {
    try {
      QByteArray result = stdCert.sign(crl, stdKey, defaults, ExtensionList(), QSsl::Der);
      QFAIL("Expected exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCrl);
      QCOMPARE(e.failure(), SigningException::NoCACertificate);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testCRLSignTimeRestraints()
  {
    Certificate::SignatureDetails details(defaults);
    details.expiryDate = QDateTime::currentDateTime().addSecs(-30);
    try {
      QByteArray result = caCert.sign(crl, caKey, details, ExtensionList(), QSsl::Der);
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCrl);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }

    details.expiryDate = caCert.expiryDate().addSecs(1);
    try {
      QByteArray result = caCert.sign(crl, caKey, details, ExtensionList(), QSsl::Der);
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCrl);
      QCOMPARE(e.failure(), SigningException::TimeConstraint);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testCRLSignKeyRestraints()
  {
    try {
      QByteArray result = caCert.sign(crl, QSslKey(), defaults, ExtensionList(), QSsl::Der);
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCrl);
      QCOMPARE(e.failure(), SigningException::KeyMismatch);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }

    try {
      QByteArray result = caCert.sign(crl, stdKey, defaults, ExtensionList(), QSsl::Der);
      QFAIL("Expected Exception not thrown.");
    }
    catch (SigningException& e) {
      QCOMPARE(e.operation(), SigningException::SignCrl);
      QCOMPARE(e.failure(), SigningException::KeyMismatch);
      QVERIFY(e.what());
    }
    catch (OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void testCRLSigning()
  {
    try {
      QByteArray pem = caCert.sign(crl, caKey, defaults, ExtensionList(), QSsl::Pem);
      QByteArray der = caCert.sign(crl, caKey, defaults, ExtensionList(), QSsl::Der);

      QVERIFY(!pem.isEmpty());
      QVERIFY(pem.startsWith("-----BEGIN X509 CRL-----\n"));
      QVERIFY(pem.endsWith("-----END X509 CRL-----\n"));

      QVERIFY(!der.isEmpty());
    }
    catch(SigningException& e) {
      QFAIL(QString("Signing exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
    catch(OpenSSLException& e) {
      QFAIL(QString("OpenSSL exception: %1 @ %2").arg(e.what()).arg(e.where()).toUtf8());
    }
  }

  void cleanupTestCase()
  {
    caKey.clear();
    requestKey.clear();
    stdKey.clear();

    caCert.clear();
    stdCert.clear();

    subject.clear();

    crl.clear();
  }

private:
  QByteArray subject;

  QSslKey caKey, requestKey, stdKey;
  Certificate caCert, stdCert;

  Certificate::SignatureDetails defaults;

  CRL crl;

  Extension::ObjectID customOID;

  Request request;
};  /* End class CertificateTest */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

QTEST_MAIN(Kca::OpenSSL::CertificateTest);
#include "certificatetest.moc"
