/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _KCAOSSL_REQUEST_H
#define _KCAOSSL_REQUEST_H

#include "export.h"
#include "common.h"
#include "extension.h"

#include <QtCore/QByteArray>

#include <QtNetwork/QSsl>

struct X509_req_st;
typedef X509_req_st X509_REQ;

namespace Kca {
namespace OpenSSL {

class Certificate;
class Key;

/**
 * Basic certificate signing request handler, providing access to request information.
 *
 * @author Felix Tiede <info@pc-tiede.de>
 * @short CSR handling class.
 */
class _KCAOSSL_EXPORT Request {

  _KCAOSSL_UNIT_TESTABLE
  friend class Certificate;

  /** Default constructor from byte array, supporting PEM (Base64) and DER (binary) encoding. */
  Request(const QByteArray& data = QByteArray(), QSsl::EncodingFormat format = QSsl::Pem);

  /** Default copy constructor. */
  Request(const Request& other);

  /** Default destructor. */
  ~Request();

  /**
   * Returns true if this is a null request (i.e., a request with no content);
   * otherwise returns false.
   */
  bool isNull() const;

  /** Returns the subject of the request. */
  QByteArray subject() const;

  /** Get X.509 v3 extensions of this request. */
  ExtensionList extensions() const;

  /** Returns this request converted to a DER (binary) encoded representation. */
  QByteArray toDer() const;
  /** Returns this certificate converted to a PEM (Base64) encoded representation. */
  QByteArray toPem() const;

  /** Returns the request's version string. */
  QByteArray version() const;

  /** Returns true if this request is not the same as other, otherwise returns false. */
  bool operator!=(const Request& other) const;
  /** Returns true if this request is the same as other, otherwise returns false. */
  bool operator==(const Request& other) const;

  /** Assignment operator. */
  Request& operator=(const Request& other);

  /**
   * Generates a request using the given key and information.
   * @param key RSA key-pair of which the public key requests signature.
   * @param subject Subject of public key part of request.
   * @param extensions X.509 v3 extensions to be added to request, defaults to empty list.
   * @param digest Digest method to sign the request with the private key part of key-pair.
   *
   * @return Generated certificate signing request or empty request if an error occurred.
   *
   * @throw OpenSSLException if a low-level OpenSSL error occurred during request generation.
   */
  static Request generate(const Key& key, const QByteArray& subject,
                          const ExtensionList& extensions = ExtensionList(),
                          Digest digest = SHA256);

protected:
  /** Get OpenSSL's representation of this request. Handle must not be free()'d. */
  X509_REQ* handle() const;

private:
  class Private;
  Private * const d;
};  /* End class Request */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

#endif  /* _KCAOSSL_REQUEST_H */
