/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "request.h"

#include "common.h"
#include "key.h"
#include "utils.h"

#include <QtCore/QByteArray>
#include <QtCore/QObject>

#include <QtNetwork/QSsl>

#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/safestack.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

namespace Kca {
namespace OpenSSL {

class Request::Private {
public:
  Private() :
      subject(QByteArray()),
      version(QByteArray()),
      extensions(ExtensionList()),
      x509_req(NULL)
  {
  }

  Private(const Private& other) :
      subject(other.subject),
      version(other.version),
      extensions(other.extensions),
      x509_req(NULL)
  {
    if (other.x509_req)
      x509_req = X509_REQ_dup(other.x509_req);
  }

  ~Private()
  {
    if (x509_req)
      X509_REQ_free(x509_req);
  }

  Private& operator=(const Private& other)
  {
    subject = other.subject;
    version = other.version;
    extensions = other.extensions;

    if (x509_req) {
      X509_REQ_free(x509_req);
      x509_req = NULL;
    }
    if (other.x509_req)
      x509_req = X509_REQ_dup(other.x509_req);

    return *this;
  }

  QByteArray subject;
  QByteArray version;

  ExtensionList extensions;

  X509_REQ* x509_req;
};  /* End class Request::Private */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

using namespace Kca::OpenSSL;

Request::Request(const QByteArray& data, QSsl::EncodingFormat format) :
    d(new Private)
{
  if (data.isNull() || data.isEmpty())
    return;

  d->x509_req = X509_REQ_new();
  if (!d->x509_req)
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

  BIO *memory = BIO_new_mem_buf((void*) data.data(), data.length());
  if (!memory) {
    X509_REQ_free(d->x509_req);
    d->x509_req = NULL;
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  switch (format) {
    case QSsl::Der:
      if (!d2i_X509_REQ_bio(memory, &d->x509_req)) {
        BIO_free(memory);
        X509_REQ_free(d->x509_req);
        d->x509_req = NULL;
        throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
      }
      break;
    case QSsl::Pem:
      if (!PEM_read_bio_X509_REQ(memory, &d->x509_req, NULL, NULL)) {
        BIO_free(memory);
        X509_REQ_free(d->x509_req);
        d->x509_req = NULL;
        throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
      }
      break;
    default:
      BIO_free(memory);
      X509_REQ_free(d->x509_req);
      d->x509_req = NULL;
      throw OpenSSLException("Unsupported encoding format", __PRETTY_FUNCTION__, __LINE__);
      break;
  }

  BIO_free(memory);

  char *subject = X509_NAME_oneline(d->x509_req->req_info->subject, NULL, 0);
  d->subject = QByteArray(subject);
  free(subject);

  d->version = QByteArray().number((qulonglong) ASN1_INTEGER_get(d->x509_req->req_info->version), 16);

  STACK_OF(X509_EXTENSION) *exts = X509_REQ_get_extensions(d->x509_req);
  if (exts) {
    d->extensions = Utils::extensions(exts);
    sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
  }
}

Request::Request(const Request& other) :
    d(new Private(*(other.d)))
{
}

Request::~Request()
{
  delete d;
}

bool Request::isNull() const
{
  return (d->x509_req == NULL);
}

QByteArray Request::subject() const
{
  return d->subject;
}

ExtensionList Request::extensions() const
{
  return d->extensions;
}

QByteArray Request::toDer() const
{
  if (!d->x509_req)
    return QByteArray();

  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *bptr;
  if (!memory)
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

  if (!i2d_X509_REQ_bio(memory, d->x509_req)) {
    BIO_free(memory);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  BIO_get_mem_ptr(memory, &bptr);
  QByteArray result(bptr->data, bptr->length);
  BIO_free(memory);

  return result;
}

QByteArray Request::toPem() const
{
  if (!d->x509_req)
    return QByteArray();

  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *bptr;
  if (!memory)
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

  if (!PEM_write_bio_X509_REQ(memory, d->x509_req)) {
    BIO_free(memory);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  BIO_get_mem_ptr(memory, &bptr);
  QByteArray result(bptr->data, bptr->length);
  BIO_free(memory);

  return result;
}

QByteArray Request::version() const
{
  return d->version;
}

bool Request::operator!=(const Request& other) const
{
  return !operator==(other);
}

bool Request::operator==(const Request& other) const
{
  // A bit ugly, but for further checks it must be ensured that this and other are both not empty.
  if (this->isNull())
    if (other.isNull())
      return true;
    else
      return false;
  else
    if (other.isNull())
      return false;

  if (d->extensions != other.d->extensions)
    return false;

  int result;
  result = X509_NAME_cmp(d->x509_req->req_info->subject, other.d->x509_req->req_info->subject);
  if (result != 0)
    return false;

  EVP_PKEY *local = X509_REQ_get_pubkey(d->x509_req);
  EVP_PKEY *operand = X509_REQ_get_pubkey(other.d->x509_req);
  result = EVP_PKEY_cmp(local, operand);
  EVP_PKEY_free(operand);
  EVP_PKEY_free(local);
  if (result != 1)
    return false;

  if (sk_X509_ATTRIBUTE_num(d->x509_req->req_info->attributes)
      != sk_X509_ATTRIBUTE_num(other.d->x509_req->req_info->attributes))
    return false;

  return true;
}

Request& Request::operator=(const Request& other)
{
  // Check for self-assignment.
  if (this == &other)
    return *this;

  *d = *(other.d);

  return *this;
}

X509_REQ* Request::handle() const
{
  return X509_REQ_dup(d->x509_req);
}

Request Request::generate(const Key& key, const QByteArray& subject,
                          const ExtensionList& extensions, Digest digest)
{
  if (key.type() != QSsl::PrivateKey)
    return Request();

  X509_REQ *request = X509_REQ_new();
  if (!request)
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);


  X509_NAME *name = Utils::x509name(subject);
  if (!name) {
    X509_REQ_free(request);
    return Request();
  }

  X509_REQ_set_version(request, 2L);
  X509_REQ_set_subject_name(request, name);

  X509_NAME_free(name);

  EVP_PKEY *sslKey = key.handle();
  if (!sslKey) {
    X509_REQ_free(request);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  if (X509_REQ_set_pubkey(request, sslKey)) {
    if (extensions.length() > 0) {
      STACK_OF(X509_EXTENSION) *exts = sk_X509_EXTENSION_new_null();
      if (exts) {
        X509V3_CTX v3_ctx;
        X509V3_set_ctx(&v3_ctx, NULL, NULL, request, NULL, 0);

        X509_EXTENSION *ext;
        for (ExtensionList::const_iterator it = extensions.constBegin();
            it != extensions.constEnd(); ++it) {
          try {
            ext = ((*it)).handle(&v3_ctx);
          }
          catch (OpenSSLException& e) {
            sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
            EVP_PKEY_free(sslKey);
            X509_REQ_free(request);
            throw;
          }
          if (ext) {
            sk_X509_EXTENSION_push(exts, ext);
          }
        }
        X509_REQ_add_extensions(request, exts);
        sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
      }
    }

    if (!X509_REQ_sign(request, sslKey, Utils::digest(digest))) {
      EVP_PKEY_free(sslKey);
      X509_REQ_free(request);
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }
    EVP_PKEY_free(sslKey);

    Request result;
    result.d->subject = subject;
    result.d->version = QByteArray().number((qulonglong) ASN1_INTEGER_get(request->req_info->version), 16);
    result.d->extensions = extensions;
    result.d->x509_req = request;

    return result;
  }

  EVP_PKEY_free(sslKey);
  X509_REQ_free(request);
  throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
}
