/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _KCAOSSL_KEY_H
#define _KCAOSSL_KEY_H

#include "export.h"

#include <QtNetwork/QSsl>
#include <QtNetwork/QSslKey>

struct evp_pkey_st;
typedef evp_pkey_st EVP_PKEY;

namespace Kca {
namespace OpenSSL {

class Certificate;
class Request;

/**
 * Extension to QSslKey providing a key-pair generator and easy access
 * to OpenSSL key handle.
 *
 * @author Felix Tiede <info@pc-tiede.de>
 * @short Extends QSslKey for easier handling in the library.
 */
class _KCAOSSL_EXPORT Key : public QSslKey {

  _KCAOSSL_UNIT_TESTABLE
  friend class Certificate;
  friend class Request;

public:
  /**
   * Default constructor from any QSslKey.
   *
   * @param key QSslKey on which this instance is based.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   */
  Key(const QSslKey& key = QSslKey());
  /**
   * Copy constructor.
   *
   * @param other Key of which this instance is a copy.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   */
  Key(const Key& other);

  /** Default destructor. */
  ~Key();

  /**
   * Assignment operator.
   *
   * @param other Key which is copied into this instance.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   */
  Key& operator=(const Key& other);

  /**
   * Generate a new key-pair using OpenSSL's key generation system with the given length in bit.
   *
   * @param length Key-length in bit, defaults to 2048.
   * @param algorithm Algorithm of key to generate, DSA or RSA, currently only RSA is supported.
   * @return Generated key-pair.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   */
  static const Key generateKeyPair(int length = 2048, QSsl::KeyAlgorithm algorithm = QSsl::Rsa);

protected:
  /** Get OpenSSL representation of the instance's key. */
  EVP_PKEY* handle() const;

private:
  class Private;
  Private * const e;
};  /* End class Key */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

#endif  /* _KCAOSSL_KEY_H */
