/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _KCAOSSL_EXTENSION_H
#define _KCAOSSL_EXTENSION_H

#include "export.h"
#include "common.h"
#include "opensslexception.h"

#include <QtCore/QString>

struct v3_ext_ctx;
typedef v3_ext_ctx X509V3_CTX;
struct X509_extension_st;
typedef X509_extension_st X509_EXTENSION;

namespace Kca {
namespace OpenSSL {

class Certificate;
class Request;
class Utils;

/**
 * Wrapper class to handle X.509 extensions.
 *
 * @author Felix Tiede <infor@pc-tiede.de>
 * @short X.509 Extension wrapper
 */
class _KCAOSSL_EXPORT Extension {

  _KCAOSSL_UNIT_TESTABLE
  friend class Certificate;
  friend class Request;
  friend class Utils;

public:
  /** Struct to carry an OID along with its long and short names. */
  struct ObjectID {
    QString oid;        /**<Object IDentifier, numerical form, might be empty. */
    QString longName;   /**<Long name of OID */
    QString shortName;  /**<Short name of OID */
  };  /* End struct ObjectID */

  /** Default constructor of extension. */
  Extension(const ObjectID& oid, const QString& value, bool critical = false, bool replace = false);
  /** Default copy constructor. */
  Extension(const Extension& other);
  /** Default destructor. */
  ~Extension();

  /** Get Object ID of extension. */
  const ObjectID oid() const;

  /** Get value of extension. */
  const QString value() const;
  /** Set new value of extension. */
  void setValue(const QString& value);

  /** Get criticality of extension. */
  bool critical() const;
  /** Set criticality of extension. */
  void setCritical(bool critical);

  /** Returns true if this extension replaces another extension with the same name. */
  bool replace() const;
  /** Set replacement flag of extension. */
  void setReplace(bool replace);

  /** Comparison operator. */
  bool operator==(const Extension& other) const;
  /** Mismatch comparison operator. */
  bool operator!=(const Extension& other) const;

  /** Assignment operator. */
  Extension& operator=(const Extension& other);

protected:
  /** Constructor with name lookup. */
  Extension(const QString& name, const QString& value,
            bool critical = false, bool replace = false);
  /** Constructor with nid lookup. */
  Extension(int nid, const QString& value,
            bool critical = false, bool replace = false);
  /** Constructor from OpenSSL's representation of extension. */
  Extension(const X509_EXTENSION* handle);
  /** Get a pointer to OpenSSL's representation of this extension. */
  X509_EXTENSION* handle(X509V3_CTX* ctx = NULL) const;

private:
  class Private;
  Private * const d;
};  /* End class Extension */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

#endif  /* _KCAOSSL_EXTENSION_H */
