/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "extension.h"

#include "opensslexception.h"
#include "utils.h"

#include <QtCore/QObject>

#include <openssl/err.h>
#include <openssl/objects.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#include <stdlib.h>

namespace Kca {
namespace OpenSSL {

class Extension::Private {
public:
  int nid;
  bool critical, replace;

  Extension::ObjectID oid;
  QString value;

  /** Assignment operator. */
  Private& operator=(const Private& other)
  {
    this->nid = other.nid;
    this->critical = other.critical;
    this->replace  = other.replace;

    this->oid.oid       = other.oid.oid;
    this->oid.longName  = other.oid.longName;
    this->oid.shortName = other.oid.shortName;
    this->value = other.value;

    return *this;
  }

  static Extension::ObjectID objectId(const ASN1_OBJECT *object)
  {
    Extension::ObjectID result;
    size_t len, bufsize = 80;
    char *buffer = NULL;

    // Adapt buffer to the size required by OBJ_obj2txt(). Should be run only once.
    do {
      buffer =(char *) realloc(buffer, bufsize);
      if (buffer)
        len = OBJ_obj2txt(buffer, bufsize, object, 1);
      else
        throw OpenSSLException("Could not allocate memory for string transfer.", __PRETTY_FUNCTION__, __LINE__);
    }
    while (len > bufsize);

    result.oid = QString(buffer);
    free(buffer);

    if (len <= 0)
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

    result.shortName = OBJ_nid2sn(OBJ_obj2nid(object));
    result.longName  = OBJ_nid2ln(OBJ_obj2nid(object));

    return result;
  }
};  /* End class Extension::Private */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

using namespace Kca::OpenSSL;

Extension::Extension(const Extension::ObjectID& oid, const QString& value,
                     bool critical, bool replace) :
    d(new Private)
{
  d->oid   = oid;
  d->value = value;

  d->critical = critical;
  d->replace  = replace;

  QString id = oid.oid;
  if (id.isEmpty())
    id = oid.shortName;

  int nid = OBJ_txt2nid(id.toLocal8Bit().constData());
  if (nid == NID_undef) {
    nid = OBJ_create(oid.oid.toLocal8Bit().constData(),
                     oid.shortName.toLocal8Bit().constData(),
                     oid.longName.toLocal8Bit().constData());
    X509V3_EXT_add_alias(nid, NID_netscape_comment);
  }
  d->nid = nid;
}

Extension::Extension(const QString& name, const QString& value,
                     bool critical, bool replace) :
    d(new Private)
{
  ASN1_OBJECT *obj = OBJ_txt2obj(name.toLocal8Bit().constData(), 0);
  if (!obj) {
    delete d;
    throw OpenSSLException("Unknown object identifier.", __PRETTY_FUNCTION__, __LINE__);
  }

  d->nid = OBJ_obj2nid(obj);
  d->oid = Private::objectId(obj);

  d->value = value;

  d->critical = critical;
  d->replace  = replace;
}

Extension::Extension(int nid, const QString& value,
                     bool critical, bool replace) :
    d(new Private)
{
  if (nid != NID_undef) {
    ASN1_OBJECT *obj = OBJ_nid2obj(nid);
    if (!obj) {
      delete d;
      throw OpenSSLException("Undefined numerical object identifier.", __PRETTY_FUNCTION__, __LINE__);
    }

    d->nid   = nid;
    d->oid = Private::objectId(obj);
    ASN1_OBJECT_free(obj);
  }
  else {
    delete d;
    throw OpenSSLException("Undefined numerical object identifier.", __PRETTY_FUNCTION__, __LINE__);
  }

  d->value = value;

  d->critical = critical;
  d->replace  = replace;
}

Extension::Extension(const Extension& other) : d(new Private)
{
  *(this->d) = *(other.d);
}

Extension::Extension(const X509_EXTENSION* handle) : d(new Private)
{
  if (!handle) {
    delete d;
    throw OpenSSLException("Need a valid pointer to a X509_EXTENSION.", __PRETTY_FUNCTION__, __LINE__);
  }

  d->oid = Private::objectId(handle->object);
  d->nid = handle->object->nid;
  if (d->nid == NID_undef) {
    d->nid = OBJ_create(d->oid.oid.toLocal8Bit().constData(),
                        d->oid.shortName.toLocal8Bit().constData(),
                        d->oid.longName.toLocal8Bit().constData());
    X509V3_EXT_add_alias(d->nid, NID_netscape_comment);
  }

  d->critical = X509_EXTENSION_get_critical((X509_EXTENSION *) handle);

  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *bptr;
  if (!memory) {
    delete d;
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  if (!X509V3_EXT_print(memory,(X509_EXTENSION *) handle, 0, 0)) {
    M_ASN1_OCTET_STRING_print(memory, handle->value);
  }

  if (BIO_get_mem_ptr(memory, &bptr))
    d->value = QString(QByteArray(bptr->data, bptr->length));
  else {
    BIO_free(memory);
    delete d;
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  BIO_free(memory);
}

Extension::~Extension()
{
  delete d;
}

const Extension::ObjectID Extension::oid() const
{
  return d->oid;
}

const QString Extension::value() const
{
  return d->value;
}
void Extension::setValue(const QString& value)
{
  d->value = value;
}

bool Extension::critical() const
{
  return d->critical;
}
void Extension::setCritical(bool critical)
{
  d->critical = critical;
}

bool Extension::replace() const
{
  return d->replace;
}
void Extension::setReplace(bool replace)
{
  d->replace = replace;
}

bool Extension::operator==(const Extension& other) const
{
  return ((d->critical == other.d->critical) &&
          ((d->nid == other.d->nid) || (d->oid.oid == other.d->oid.oid)) &&
          (d->value == other.d->value));
}

bool Extension::operator!=(const Extension& other) const
{
  return ((d->critical != other.d->critical) || (d->nid != other.d->nid) || (d->value != other.d->value));
}

Extension& Extension::operator=(const Extension& other)
{
  // Check for self-assignment.
  if (this == &other)
    return *this;

  *(this->d) = *(other.d);
  return *this;
}

X509_EXTENSION* Extension::handle(X509V3_CTX* ctx) const
{
  X509_EXTENSION *result = NULL;
  if (ctx) {
    result = X509V3_EXT_conf_nid(NULL, ctx, d->nid, d->value.toLocal8Bit().data());
    if (result) {
      // X.509 extensions start with "0x16 0x0a" if created as above and value needs to be set again.
      if (QByteArray((char *) result->value->data).startsWith(QByteArray::fromHex("160a"))) {
        ASN1_OCTET_STRING *value = Utils::octetString(d->value);
        if (value) {
          X509_EXTENSION_set_data(result, value);
          ASN1_OCTET_STRING_free(value);
        }
      }
      if (!X509_EXTENSION_set_critical(result, d->critical))
        throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

      return result;
    }

    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }
  else {
    ASN1_OCTET_STRING *value = Utils::octetString(d->value);
    if (value) {
      result = X509_EXTENSION_create_by_NID(NULL, d->nid, d->critical, value);
      ASN1_OCTET_STRING_free(value);
      if (!result) {
        throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
      }
    }

    return result;
  }
}
