/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _KCAOSSL_CERTIFICATE_H
#define _KCAOSSL_CERTIFICATE_H

#include "export.h"
#include "common.h"

#include <QtCore/QDateTime>
#include <QtCore/QtGlobal>

#include <QtNetwork/QSsl>
#include <QtNetwork/QSslCertificate>

class QString;

namespace Kca {
namespace OpenSSL {

class Key;
class Request;

/**
 * Extension to QSslCertificate providing signing methods.
 *
 * @author Felix Tiede <info@pc-tiede.de>
 * @short Signing-capable QSslCertificate extension.
 */
class _KCAOSSL_EXPORT Certificate : public QSslCertificate {

  _KCAOSSL_UNIT_TESTABLE

public:
  /**
   * Details of signature, common for all signing operations.<br>
   * Signing operations imply some constraints not enforced by this struct: A serial may not be
   * 0 (zero) or ~0 and expiryDate must be after effectiveDate. If those constraints are not met
   * the signing operation will fail and throw a SigningException.
   *
   * @author Felix Tiede <info@pc-tiede.de>
   * @short Details for signing operations.
   */
  struct SignatureDetails {
    quint64 serial;           /**<Serial number to set for signed object. */
    Digest digest;            /**<Cryptographic digest to use on signing operation. */
    QDateTime effectiveDate;  /**<Timestamp when signed object starts to be valid. */
    QDateTime expiryDate;     /**<Timestamp when signed object expires. */
  };  /* End struct SignatureDetails */

  /** Default constructor from ancestor class. */
  Certificate(const QSslCertificate& certificate = QSslCertificate());

  /** Default copy constructor. */
  Certificate(const Certificate& other);

  /**
   * Generate a self-signed certificate.
   *
   * @param key Private key used for self-signing.
   * @param subject Subject info string of self-signed certificate.
   * @param details Details of self-generated signature.
   * @param extensions Extensions of generated certificate.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   * @throw SigningException Thrown when a problem with the given details or extensions occurs.
   */
  Certificate(const Key& key, const QByteArray& subject,
              const SignatureDetails& details, const ExtensionList& extensions);

  /** Default destructor. */
  ~Certificate();

  /**
   * Check if certificate is a CA certificate and can sign.
   *
   * @return true if the certificate is a CA certificate, otherwise false.
   */
  bool isCA() const;

  /**
   * Checks if the given private key matches this certificate instance.
   *
   * @param key Key to check against this certificate instance.
   * @return true if the given key is a private key and matches, otherwise false.
   */
  bool keyMatch(const Key& key) const;

  /** Get X.509 v3 extensions of this certificate. */
  ExtensionList extensions() const;

  /**
   * Sign a certificate signing request and generate a certificate.
   *
   * @param request Request to be signed.
   * @param signingKey Private key of this instance to perform signing operation.
   * @param details Details of signing operation.
   * @param extensions List of extensions to add to new certificate.
   * @return Signed certificate or empty if signing operation failed.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   * @throw SigningException Thrown when a problem with the given details or extensions occurs.
   */
  Certificate sign(const Request& request, const Key& signingKey,
                   const SignatureDetails& details,
                   const ExtensionList& extensions) const;

  /**
   * Sign a certificate revocation list.
   *
   * @param crl Certificate revocation list to be signed.
   * @param signingKey Private key of this instance to perform signing operation.
   * @param details Details of signing operation.
   * @param extensions List of extensions to add to new CRL.
   * @param format Encoding of signed CRL in the byte array, defaults to DER.
   * @return Signed CRL or empty if signing operation failed.
   *
   * @throw OpenSSLException Thrown when a low-level problem with the OpenSSL library occurs.
   * @throw SigningException Thrown when a problem with the given details or extensions occurs.
   */
  QByteArray sign(const CRL& crl, const Key& signingKey,
                  const SignatureDetails& details,
                  const ExtensionList& extensions, QSsl::EncodingFormat format = QSsl::Der) const;

  /** Assignment operator. */
  Certificate& operator=(const Certificate& other);

private:
  class Private;
  Private * const e;
};  /* End class Certificate */

};  /* End namespace OpenSSL */
};  /* End namespace Kca */

#endif  /* _KCAOSSL_CERTIFICATE_H */
