/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "certificate.h"

#include "extension.h"
#include "key.h"
#include "opensslexception.h"
#include "signingexception.h"
#include "request.h"
#include "utils.h"

#include <QtCore/QByteArray>

#include <QtNetwork/QSsl>
#include <QtNetwork/QSslCertificate>

#include <openssl/asn1.h>
#include <openssl/bn.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

namespace Kca
{
namespace OpenSSL
{
class Certificate::Private
{
public:
  /** Copy constructor. */
  Private(const Private& other) :
    ca(other.ca),
    extensions(other.extensions),
    x509(NULL)
  {
    if (other.x509)
      x509 = X509_dup(other.x509);
  };

  /** Initialization constructor from QSslCertificate. */
  Private(const QSslCertificate& certificate = QSslCertificate()) :
    ca(false),
    extensions(ExtensionList()),
    x509(NULL)
  {
    if (certificate.isNull())
      return;

    x509 = X509_new();
    if (!x509)
      return;

    QByteArray derBytes = certificate.toDer();
    const unsigned char *der =(const unsigned char *) derBytes.constData();

    if (!d2i_X509(&x509, &der, derBytes.length())) {
      delete [] der;
      derBytes.clear();
      X509_free(x509);
      x509 = NULL;

      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }

    extensions = Utils::extensions(x509->cert_info->extensions);

    int loc_basicConstraints = X509_get_ext_by_NID(x509, NID_basic_constraints,
                                                   X509_get_ext_count(x509)-1);
    if (loc_basicConstraints < 0)
      return;

    unsigned char *buffer;
    X509_EXTENSION *basicConstraints = X509_get_ext(x509, loc_basicConstraints);
    if (!basicConstraints)
      return;

    size_t len = ASN1_STRING_to_UTF8(&buffer, basicConstraints->value);
    if (len > 0) {
      QByteArray value((const char *) buffer, len);
      ca = (X509_EXTENSION_get_critical(basicConstraints) && value.toLower().contains("ca:true"));
      value.clear();
      OPENSSL_free(buffer);
    }
    X509_EXTENSION_free(basicConstraints);
  };

  /** Default destructor. */
  ~Private()
  {
    if (x509)
      X509_free(x509);
  };

  /** Assignment operator. */
  Private& operator=(const Private& other)
  {
    ca = other.ca;
    extensions = other.extensions;
    if (x509) {
      X509_free(x509);
      x509 = NULL;
    }
    if (other.x509)
      x509 = X509_dup(other.x509);

    return *this;
  };

  /** Convert a 64 bit integer into an OpenSSL integer. */
  static ASN1_INTEGER* convert(quint64 in)
  {
    if (in ==(quint64) ~0 || in ==(quint64) 0)
      throw OpenSSLException("Invalid serial provided.", __PRETTY_FUNCTION__, __LINE__);

    ASN1_INTEGER *result = ASN1_INTEGER_new();
    if (!result)
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

    BIGNUM *bn = BN_new();
    if (!bn) {
      ASN1_INTEGER_free(result);
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }

    if (!BN_hex2bn(&bn, QString().setNum(in, 16).toLocal8Bit().constData())) {
      BN_free(bn);
      ASN1_INTEGER_free(result);
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }
    BN_to_ASN1_INTEGER(bn, result);
    BN_free(bn);

    return result;
  };

  /** Create a X.509 structure and populate it with given values for signing. */
  static X509* createX509(const EVP_PKEY* publicKey, const QByteArray& subject,
                          const Certificate::SignatureDetails& details,
                          const ExtensionList& extensions, const X509* issuer = NULL, bool* ca = NULL)
  {
    X509* x509 = X509_new();
    if (!x509)
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);


    ASN1_INTEGER *serial = Private::convert(details.serial);
    if (!serial) {
      X509_free(x509);
      throw SigningException(SigningException::SignCsr, SigningException::SerialConstraint,
                             "Provided serial can not be used.", __PRETTY_FUNCTION__, __LINE__);
    }

    // Basic certificate setup.
    X509_set_version(x509, 2);
    X509_set_serialNumber(x509, serial);
    ASN1_INTEGER_free(serial);

    X509_set_pubkey(x509,(EVP_PKEY*) publicKey);

    // Set subject and issuer name.
    X509_NAME *name = Utils::x509name(subject);
    if (!name) {
      X509_free(x509);
      throw OpenSSLException("Subject can not be added to certificate.", __PRETTY_FUNCTION__, __LINE__);
    }

    X509_set_subject_name(x509, name);
    X509_set_issuer_name(x509, X509_get_subject_name((issuer ?(X509*) issuer : x509)));
    X509_NAME_free(name);

    // Set validity timespan.
    ASN1_TIME *tm = ASN1_TIME_new();
    if (!tm) {
      X509_free(x509);
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }
    ASN1_TIME_set(tm, details.effectiveDate.toUTC().toTime_t());
    X509_set_notBefore(x509, tm);
    ASN1_TIME_set(tm, details.expiryDate.toUTC().toTime_t());
    X509_set_notAfter(x509, tm);
    ASN1_TIME_free(tm);

    // Add extensions to certificate.
    X509V3_CTX v3ctx;
    X509V3_set_ctx_nodb(&v3ctx);
    X509V3_set_ctx(&v3ctx, (issuer ?(X509*) issuer : x509), x509, NULL, NULL, 0);
    X509_EXTENSION *ext;
    for (ExtensionList::const_iterator it = extensions.constBegin();
        it != extensions.constEnd(); ++it) {
      try {
        ext = (*it).handle(&v3ctx);
      }
      catch (OpenSSLException& e) {
        X509_free(x509);
        throw;
      }
      if (ext) {
        X509_add_ext(x509, ext, -1);
        X509_EXTENSION_free(ext);
      }
      if (ca)
        *ca = (*ca || ((*it).critical() && (*it).oid().oid == "2.5.29.19" &&
               (*it).value().toLower().contains("ca:true")));
    }

    return x509;
  };

  /**
   * Local property storing whether this instance is a CA or not.
   * Since a certificate never changes this property can be populated on instantiation and speeds
   * up isCA().
   */
  bool ca;

  /** List of certificate's extensions. */
  ExtensionList extensions;

  /** Local OpenSSL representation of local certificate, populated on instantiation. */
  X509* x509;
};  // End class Certificate::Private

};  // End namespace OpenSSL
};  // End namespace Kca

using namespace Kca::OpenSSL;

Certificate::Certificate(const QSslCertificate& certificate) :
    QSslCertificate(certificate),
    e(new Private(certificate))
{
}

Certificate::Certificate(const Certificate& other) :
    QSslCertificate(other),
    e(new Private(*(other.e)))
{
}

Certificate::Certificate(const Key& key, const QByteArray& subject,
                         const Certificate::SignatureDetails& details,
                         const ExtensionList& extensions) :
    QSslCertificate(),
    e(new Private)
{
  if (key.isNull() || key.algorithm() != QSsl::Rsa || key.type() != QSsl::PrivateKey)
    throw SigningException(SigningException::SignCsr, SigningException::KeyMismatch,
                           "Supplied key can not generate certificate.", __PRETTY_FUNCTION__, __LINE__);

  if (details.effectiveDate > details.expiryDate || details.expiryDate < QDateTime::currentDateTime())
    throw SigningException(SigningException::SignCsr, SigningException::TimeConstraint,
                           "Supplied validity timestamps can not be used.", __PRETTY_FUNCTION__, __LINE__);

  EVP_PKEY *pkey = key.handle();
  if (!pkey)
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);

  try {
    e->x509 = Private::createX509(pkey, subject, details, extensions, NULL, &e->ca);
  }
  catch (OpenSSLException& e) {
    EVP_PKEY_free(pkey);
    throw;
  }

  // Finally sign new certificate.
  if (!X509_sign(e->x509, pkey, Utils::digest(details.digest))) {
    X509_free(e->x509);
    e->x509 = NULL;
    EVP_PKEY_free(pkey);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  EVP_PKEY_free(pkey);

  e->extensions = Utils::extensions(e->x509->cert_info->extensions);

  // Transfer certificate to superclass
  BIO *bio = BIO_new(BIO_s_mem());
  BUF_MEM *bptr;
  if (!bio) {
    X509_free(e->x509);
    e->x509 = NULL;
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }
  if (!i2d_X509_bio(bio, e->x509)) {
    BIO_free(bio);
    X509_free(e->x509);
    e->x509 = NULL;
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  BIO_get_mem_ptr(bio, &bptr);
  QByteArray der(bptr->data, bptr->length);
  BIO_free(bio);

  QSslCertificate::operator=(QSslCertificate(der, QSsl::Der));
}

Certificate::~Certificate()
{
  delete e;
}

Certificate& Certificate::operator=(const Certificate& other)
{
  // Check for self-assignment.
  if (this == &other)
    return *this;

  QSslCertificate::operator=(other);
  *e = *(other.e);

  return *this;
}

bool Certificate::isCA() const
{
  return e->ca;
}

bool Certificate::keyMatch(const Key& key) const
{
  if (!e->x509)
    return false;

  if (key.isNull() || key.algorithm() != QSsl::Rsa || key.type() != QSsl::PrivateKey)
    return false;

  EVP_PKEY *pkey = key.handle();
  if (!pkey)
    return false;

  EVP_PKEY_free(pkey);

  return (bool) X509_check_private_key(e->x509, pkey);
}

ExtensionList Certificate::extensions() const
{
  return e->extensions;
}

Certificate Certificate::sign(const Request& request, const Key& signingKey,
                              const SignatureDetails& details,
                              const ExtensionList& extensions) const
{
  if (!e->ca)
    throw SigningException(SigningException::SignCsr, SigningException::NoCACertificate,
                           "This instance can not sign requests.", __PRETTY_FUNCTION__, __LINE__);

  if (details.effectiveDate > details.expiryDate || details.expiryDate < QDateTime::currentDateTime() ||
      details.effectiveDate < this->effectiveDate() || details.expiryDate > this->expiryDate())
    throw SigningException(SigningException::SignCsr, SigningException::TimeConstraint,
                           "Supplied validity timestamps can not be used.", __PRETTY_FUNCTION__, __LINE__);

  if (signingKey.isNull() || signingKey.algorithm() != QSsl::Rsa
      || signingKey.type() != QSsl::PrivateKey || !keyMatch(signingKey))
    throw SigningException(SigningException::SignCsr, SigningException::KeyMismatch,
                           "Provided key not suitable to sign request.", __PRETTY_FUNCTION__, __LINE__);

  X509_REQ *req = request.handle();
  if (!req)
    return Certificate();

  EVP_PKEY *requestKey = X509_REQ_get_pubkey(req);
  if (!requestKey) {
    X509_REQ_free(req);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  X509 *cert;
  try {
    cert = Private::createX509(requestKey, request.subject(), details, extensions, e->x509);
  }
  catch (OpenSSLException& e) {
    EVP_PKEY_free(requestKey);
    X509_REQ_free(req);
    throw;
  }
  X509_set_pubkey(cert, requestKey);
  EVP_PKEY_free(requestKey);

  EVP_PKEY *pkey = signingKey.handle();
  if (!pkey) {
    X509_free(cert);
    X509_REQ_free(req);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  if (!X509_sign(cert, pkey, Utils::digest(details.digest))) {
    EVP_PKEY_free(pkey);
    X509_free(cert);
    X509_REQ_free(req);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  EVP_PKEY_free(pkey);
  X509_REQ_free(req);

  // Transfer certificate to superclass
  BIO *bio = BIO_new(BIO_s_mem());
  BUF_MEM *bptr;
  if (!bio) {
    X509_free(cert);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }
  if (!i2d_X509_bio(bio, cert)) {
    BIO_free(bio);
    X509_free(cert);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  BIO_get_mem_ptr(bio, &bptr);
  QByteArray der(bptr->data, bptr->length);
  BIO_free(bio);

  X509_free(cert);

  return QSslCertificate(der, QSsl::Der);
}

QByteArray Certificate::sign(const CRL& crl, const Key& signingKey,
                             const SignatureDetails& details,
                             const ExtensionList& extensions, QSsl::EncodingFormat format) const
{
  if (!e->ca)
    throw SigningException(SigningException::SignCrl, SigningException::NoCACertificate,
                           "This instance can not sign revocation list.", __PRETTY_FUNCTION__, __LINE__);

  if (details.expiryDate < QDateTime::currentDateTime() || details.expiryDate > this->expiryDate())
    throw SigningException(SigningException::SignCrl, SigningException::TimeConstraint,
                           "Supplied expiration timestamp can not be used.", __PRETTY_FUNCTION__, __LINE__);

  if (signingKey.isNull() || signingKey.algorithm() != QSsl::Rsa
      || signingKey.type() != QSsl::PrivateKey || !keyMatch(signingKey))
    throw SigningException(SigningException::SignCrl, SigningException::KeyMismatch,
                           "Provided key not suitable to sign revocation list.", __PRETTY_FUNCTION__, __LINE__);

  ASN1_INTEGER *serial = Private::convert(details.serial);

  X509_CRL *list = X509_CRL_new();
  if (!list) {
    ASN1_INTEGER_free(serial);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  X509_CRL_set_version(list, 1);
  X509_CRL_set_issuer_name(list, X509_get_subject_name(e->x509));
  X509_CRL_add1_ext_i2d(list, NID_crl_number, serial, 0, 0);
  ASN1_INTEGER_free(serial);

  // Feed timestamps to CRL.
  ASN1_TIME *tm = ASN1_TIME_new();
  if (!tm) {
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  ASN1_TIME_set(tm, QDateTime::currentDateTime().toUTC().toTime_t());
  X509_CRL_set_lastUpdate(list, tm);
  ASN1_TIME_set(tm, details.expiryDate.toUTC().toTime_t());
  X509_CRL_set_nextUpdate(list, tm);
  ASN1_TIME_free(tm);

  ASN1_ENUMERATED *entryReason = ASN1_ENUMERATED_new();
  if (!entryReason) {
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  ASN1_INTEGER    *entrySerial;
  ASN1_TIME       *entryTime = ASN1_TIME_new();
  if (!entryTime) {
    ASN1_ENUMERATED_free(entryReason);
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  // Add entries to CRL.
  X509_REVOKED *entry;
  for (CRL::const_iterator it = crl.constBegin(); it != crl.constEnd(); ++it) {
    entry = X509_REVOKED_new();
    if (!entry) {
      ASN1_TIME_free(entryTime);
      ASN1_ENUMERATED_free(entryReason);
      X509_CRL_free(list);
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }

    try {
      entrySerial = Private::convert((*it).serial);
    }
    catch (OpenSSLException& e) {
      X509_REVOKED_free(entry);
      ASN1_TIME_free(entryTime);
      ASN1_ENUMERATED_free(entryReason);
      X509_CRL_free(list);
      throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
    }

    X509_REVOKED_set_serialNumber(entry, entrySerial);
    ASN1_INTEGER_free(entrySerial);

    ASN1_TIME_set(entryTime, (*it).timestamp.toUTC().toTime_t());
    X509_REVOKED_set_revocationDate(entry, entryTime);

    ASN1_ENUMERATED_set(entryReason, (*it).reason);
    X509_REVOKED_add1_ext_i2d(entry, NID_crl_reason, entryReason, 0, 0);

    X509_CRL_add0_revoked(list, entry);
  }
  ASN1_TIME_free(entryTime);
  ASN1_ENUMERATED_free(entryReason);

  // Add extensions to CRL.
  X509V3_CTX v3ctx;
  X509V3_set_ctx_nodb(&v3ctx);
  X509V3_set_ctx(&v3ctx, e->x509, NULL, NULL, list, 0);
  X509_EXTENSION *ext;
  for (ExtensionList::const_iterator it = extensions.constBegin();
       it != extensions.constEnd(); ++it) {
    try {
      ext = (*it).handle(&v3ctx);
    }
    catch (OpenSSLException& e) {
      X509_CRL_free(list);
      throw;
    }
    if (ext) {
      X509_CRL_add_ext(list, ext, -1);
      X509_EXTENSION_free(ext);
    }
  }

  EVP_PKEY *pkey = signingKey.handle();
  if (!pkey) {
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  if (!X509_CRL_sign(list, pkey, Utils::digest(details.digest))) {
    EVP_PKEY_free(pkey);
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  EVP_PKEY_free(pkey);

  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *bptr;
  if (!memory) {
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }

  int (*biowriter)(BIO*, X509_CRL*) = NULL;

  switch (format) {
    case QSsl::Der:
      biowriter = &i2d_X509_CRL_bio;
      break;
    case QSsl::Pem:
      biowriter = &PEM_write_bio_X509_CRL;
      break;
    default:
      BIO_free(memory);
      X509_CRL_free(list);
      throw OpenSSLException("Invalid export format.", __PRETTY_FUNCTION__, __LINE__);
  }

  if (!biowriter(memory, list)) {
    BIO_free(memory);
    X509_CRL_free(list);
    throw OpenSSLException(ERR_error_string(ERR_get_error(), NULL), __PRETTY_FUNCTION__, __LINE__);
  }
  X509_CRL_free(list);

  BIO_get_mem_ptr(memory, &bptr);
  QByteArray result(bptr->data, bptr->length);
  BIO_free(memory);

  return result;
}
