/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2007 - 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "kcaview.h"
#include "settings.h"

#include <QtCore/QList>
#include <QtCore/QString>
#include <QtCore/QStringList>

#include <QtGui/QLabel>
#include <QtGui/QTreeWidgetItem>

#include <klocale.h>

kcaView::kcaView(QWidget *)
{
    setupUi(this);
    settingsChanged();
    setAutoFillBackground(true);

    QAction *oneCrl = new QAction(i18n("... for selected authority"), this);
    oneCrl->setStatusTip(i18n("Generate CRL for selected certification authority."));
    connect(oneCrl, SIGNAL(triggered(bool)), SIGNAL(generateCrl()));

    btnGenerateCRL->addAction(oneCrl);
    connect(btnGenerateCRL, SIGNAL(clicked(bool)), SIGNAL(generateCrl()));

    connect(authoritiesTree, SIGNAL(currentItemChanged(QTreeWidgetItem*,QTreeWidgetItem*)),
            SLOT(authoritySelected(QTreeWidgetItem*)));
    connect(certificateList, SIGNAL(currentItemChanged(QTreeWidgetItem*,QTreeWidgetItem*)),
            SLOT(certificateSelected(QTreeWidgetItem*)));
}

kcaView::~kcaView()
{

}

quint32 kcaView::activeAuthority() const
{
  return m_authority;
}

quint64 kcaView::activeCertificate() const
{
  return m_certificate;
}

void kcaView::addAuthority(const quint32 parent, const quint32 hash, const QString& name,
                           const bool unlocked, const bool lockable)
{
  QTreeWidgetItem *ancestor = 0;
  QTreeWidgetItem *entry = new QTreeWidgetItem(QStringList(name), 0);
  if (!lockable)
    entry->setData(0, Qt::DecorationRole, KIcon("security-low"));
  else {
    if (unlocked)
      entry->setData(0, Qt::DecorationRole, KIcon("object-unlocked"));
    else
      entry->setData(0, Qt::DecorationRole, KIcon("security-high"));
  }

  AuthorityFlags *flags = new AuthorityFlags();
  flags->lockable = lockable;
  flags->unlocked = unlocked;
  m_authoritiesTree.insert(hash, entry);
  m_authoritiesFlags.insert(hash, flags);

  if (parent != hash)
    ancestor = m_authoritiesTree.value(parent,(QTreeWidgetItem*) 0);

  if (ancestor) {
    ancestor->addChild(entry);
    authoritiesTree->expandItem(ancestor);
  }
  else
    authoritiesTree->addTopLevelItem(entry);

  if (m_authoritiesTree.count() == 2) {
    QAction *allCRLs = new QAction(i18n("... for all authorities"), this);
    allCRLs->setStatusTip(i18n("Generate CRLs for all loaded certification authorities."));
    connect(allCRLs, SIGNAL(triggered(bool)), SIGNAL(generateAllCrls()));

    btnGenerateCRL->addAction(allCRLs);
  }
}

void kcaView::clearAuthorities()
{
  authoritiesTree->clear();
  m_authoritiesTree.clear();
}

void kcaView::showCertificates(const QList< QStringList >& certificates)
{
  m_certificate = 0;
  m_certificatesList.clear();

  certificateList->clear();
  int count = certificates.count();

  static QFont revoked;
  revoked.setStrikeOut(true);
  static QBrush redText;
  redText.setColor(Qt::red);
  static QBrush grayText;
  grayText.setColor(Qt::darkGray);

  QTreeWidgetItem *entry;
  for (int i = 0; i < count; ++i) {
    QStringList item = certificates.at(i);
    int state = item.last().toInt(0, 10);
    item.removeLast();
    entry = new QTreeWidgetItem(item, 0);
    m_certificatesList.insert(item.at(0).toULongLong(0, 16), entry);
    switch (state) {
      case 0: break;
      case 1:
        for (int i = 0; i < 6; ++i) {
          entry->setData(i, Qt::ForegroundRole, grayText);
        }
        break;
      case 2:
        for (int i = 0; i < 6; ++i) {
          entry->setData(i, Qt::FontRole, revoked);
          entry->setData(i, Qt::ForegroundRole, redText);
        }
        break;
    }
    certificateList->addTopLevelItem(entry);
  }
}

void kcaView::addCertificate(const QStringList& certificate)
{
  static QBrush greenText;
  greenText.setColor(Qt::darkGreen);

  quint64 serial = certificate.at(0).toULongLong(0, 16);
  QTreeWidgetItem *entry = new QTreeWidgetItem(certificate, 0);
  for (int i = 0; i < 6; ++i) {
    entry->setData(i, Qt::ForegroundRole, greenText);
  }
  certificateList->addTopLevelItem(entry);

  m_certificatesList.insert(serial, entry);
}

void kcaView::authorityLocked(const quint32 authority)
{
  QTreeWidgetItem *entry = m_authoritiesTree.value(authority, 0);
  if (!entry)
    return;

  entry->setData(0, Qt::DecorationRole, KIcon("security-high"));
  m_authoritiesFlags.value(authority)->unlocked = false;

  if (authority == m_authority)
    btnLock->setEnabled(false);
}

void kcaView::authorityUnlocked(const quint32 authority)
{
  QTreeWidgetItem *entry = m_authoritiesTree.value(authority, 0);
  if (!entry)
    return;

  entry->setData(0, Qt::DecorationRole, KIcon("object-unlocked"));
  m_authoritiesFlags.value(authority)->unlocked = true;

  if (authority == m_authority)
    btnLock->setEnabled(true);
}

void kcaView::authoritySelected(QTreeWidgetItem* item)
{
  m_authority = m_authoritiesTree.key(item, 0);

  btnGenerateCRL->setEnabled((m_authority));

  AuthorityFlags *flags = m_authoritiesFlags.value(m_authority, 0);
  if (flags) {
    btnLock->setEnabled(flags->lockable && flags->unlocked);
  }
  else {
    // Default GUI behavior
    btnLock->setEnabled(false);
  }

  emit authoritySelected(m_authority);
}

void kcaView::certificateSelected(QTreeWidgetItem* item)
{
  m_certificate = m_certificatesList.key(item, 0);
  pbExport->setEnabled(true);

  emit certificateSelected(m_authority, m_certificate);
}

void kcaView::switchColors()
{
    // switch the foreground/background colors of the label
    QColor color = Settings::col_background();
    Settings::setCol_background( Settings::col_foreground() );
    Settings::setCol_foreground( color );

    settingsChanged();
}

void kcaView::settingsChanged()
{
    QPalette pal;
    pal.setColor( QPalette::WindowText, Settings::col_foreground());
//    ui_kcaview_base.kcfg_sillyLabel->setPalette( pal );

    // i18n : internationalization
//    ui_kcaview_base.kcfg_sillyLabel->setText( i18n("This project is %1 days old",Settings::val_time()) );
    emit signalChangeStatusbar( i18n("Settings changed") );
}

#include "kcaview.moc"
