/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef KCADBUSINTERFACE_H
#define KCADBUSINTERFACE_H

#include "kca.h"

#include <QtDBus/QDBusAbstractAdaptor>

template< typename T > class QList;

namespace kCA {
namespace DBus {
struct Authority;

/**
 * Adaptor class for interface de.pctiede.kCA
 * This class provides a DBus interface to call most important functions
 * via DBus as well as from the GUI.
 *
 * @short DBus interface specification
 * @author Felix Tiede <info@pc-tiede.de>
 */
class Interface: public QDBusAbstractAdaptor
{
  Q_OBJECT
  Q_CLASSINFO("D-Bus Interface", "de.pctiede.kCA")
  Q_CLASSINFO("D-Bus Introspection", ""
              "  <interface name=\"de.pctiede.kCA\" >\n"
              "    <property name=\"authorities\" type=\"a(uss)\" access=\"read\">\n"
              "      <annotation name=\"com.trolltech.QtDBus.QtTypeName\" value=\"QList&lt;kCA::DBus::Authority&gt;\" />\n"
              "    </property>\n"
              "    <signal name=\"crlGenerated\" >\n"
              "      <arg type=\"u\" name=\"authority\" />\n"
              "      <arg type=\"i\" name=\"status\" />\n"
              "      <arg type=\"s\" name=\"filename\" />\n"
              "    </signal>\n"
              "    <signal name=\"certificateExported\" >\n"
              "      <arg type=\"u\" name=\"authority\" />\n"
              "      <arg type=\"t\" name=\"serial\" />\n"
              "      <arg type=\"i\" name=\"status\" />\n"
              "      <arg type=\"s\" name=\"filename\" />\n"
              "    </signal>\n"
              "    <signal name=\"certificateRevoked\" >\n"
              "      <arg type=\"u\" name=\"authority\" />\n"
              "      <arg type=\"t\" name=\"serial\" />\n"
              "      <arg type=\"i\" name=\"status\" />\n"
              "      <arg type=\"i\" name=\"reason\" />\n"
              "    </signal>\n"
              "    <signal name=\"requestSigned\" >\n"
              "      <arg type=\"s\" name=\"requestSubject\" />\n"
              "      <arg type=\"i\" name=\"status\" />\n"
              "      <arg type=\"u\" name=\"authority\" />\n"
              "      <arg type=\"t\" name=\"serial\" />\n"
              "    </signal>\n"
              "    <method name=\"exportCertificate\" >\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "      <arg direction=\"in\" type=\"t\" name=\"serial\" />\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "    </method>\n"
              "    <method name=\"verifyCertificate\" >\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "      <arg direction=\"out\" type=\"i\" />\n"
              "    </method>\n"
              "    <method name=\"verifyCertificate\" >\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "      <arg direction=\"in\" type=\"t\" name=\"serial\" />\n"
              "      <arg direction=\"out\" type=\"i\" />\n"
              "    </method>\n"
              "    <method name=\"generateCrl\" >\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "    </method>\n"
              "    <method name=\"revokeCertificate\" >\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "    </method>\n"
              "    <method name=\"revokeCertificate\" >\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "      <arg direction=\"in\" type=\"t\" name=\"serial\" />\n"
              "    </method>\n"
              "    <method name=\"revokeCertificate\" >\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "      <arg direction=\"in\" type=\"i\" name=\"reason\" />\n"
              "    </method>\n"
              "    <method name=\"revokeCertificate\" >\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "      <arg direction=\"in\" type=\"t\" name=\"serial\" />\n"
              "      <arg direction=\"in\" type=\"i\" name=\"reason\" />\n"
              "    </method>\n"
              "    <method name=\"signRequest\" >\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "    </method>\n"
              "    <method name=\"signRequest\" >\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "    </method>\n"
              "    <method name=\"signRequest\" >\n"
              "      <arg direction=\"in\" type=\"s\" name=\"filename\" />\n"
              "      <arg direction=\"in\" type=\"u\" name=\"authority\" />\n"
              "      <arg direction=\"in\" type=\"s\" name=\"notBefore\" />\n"
              "      <arg direction=\"in\" type=\"s\" name=\"notAfter\" />\n"
              "    </method>\n"
              "  </interface>\n"
              "")
  Q_PROPERTY(QList< kCA::DBus::Authority > authorities READ authorities)

  friend class ::kca;

public:
  /**
   * Creates an object to connect to DBus and exchange messages.
   * @param parent Parental object which will be responsible for dispatching DBus messages
   */
  Interface(QObject *parent);
  virtual ~Interface();

  inline kca *parent() const
  {
    return static_cast< kca* >(QObject::parent());
  };

  /** Returns a list of available certification authorities by their hash value. */
  QList< Authority > authorities();

Q_SIGNALS: // SIGNALS
  /**
   * Emitted whenever a CRL was generated and stored as a file.
   * @param authority The subject hash value of the authority issuing the CRL.
   * @param status Status of CRL generation, 0 if successful, non-0 otherwise.
   * @param filename Absolute filename of stored CRL, empty if unsuccessful.
   */
  void crlGenerated(const quint32 authority, const int status, const QString &filename);
  /**
   * Emitted whenever a certificate was exported to a file.
   * @param authority The subject hash value of the authority issuing the requested certificate.
   * @param serial The serial number of the requested certificate.
   * @param status Status of export, 0 if successful, non-0 otherwise.
   * @param filename Absolute filename of stored certificate, empty if unsuccessful.
   */
  void certificateExported(const quint32 authority, const quint64 serial, const int status, const QString &filename);
  /**
   * Emitted whenever a certificate was revoked.
   * @param authority The subject hash value of the authority issuing the certificate to be revoked.
   * @param serial The Serial number of the revoked certificate.
   * @param status Status of revocation, 0 if successful, non-0 otherwise.
   * @param reason Reason of revocation as entered into CRL
   */
  void certificateRevoked(const quint32 authority, const quint64 serial, const int status, const int reason);
  /**
   * Emitted whenever a certification request has been signed by any authority.
   * If a request file could not be read or contained no request data, requestSubject contains the filename
   * to signal which file caused the problem.
   * @param requestSubject Subject of request from which certificate was generated or given request filename.
   * @param status Status of signing, 0 if successful, non-0 otherwise.
   * @param authority The subject hash value of the authority issuing the certificate.
   * @param serial The serial number of the issued certificate.
   */
  void requestSigned(const QString &requestSubject, const int status, const quint32 authority, const quint64 serial);

public Q_SLOTS: // METHODS
  /**
   * Exports a certificate to the given filename. If the file exsists, the certificate is appended.
   * Result is reported by signal certificateExported().
   * @param authority The subject hash value of the authority issuing the requested certificate.
   * @param serial The serial number of the requested certificate.
   * @param filename File to which the certificate should be exported.
   */
  Q_SCRIPTABLE void exportCertificate(const quint32 authority, const quint64 serial, const QString &filename);

  /**
   * Generates current CRL for given authority and stores it in the given filename.
   * If the file exists, the CRL is appended. Result is reported by signal crlGenerated().
   * @param authority Subject hash value of authority to use.
   * @param filename File to which the CRL should be exported.
   */
  Q_SCRIPTABLE void generateCrl(const quint32 authority, const QString &filename);
  /**
   * Revokes a certificate. The GUI will provide reason as default value but allow changing.
   * This method returns immediately.
   * If callers need to verify success of revocation they should connect
   * to certificateRevoked().
   * @param filename Filename of certificate to be revoked.
   * @param reason Reason of revocation as specified in RevocationReasons, defaults to -1.
   */
  Q_SCRIPTABLE void revokeCertificate(const QString &filename, const int reason = 0);
  /**
   * Revokes a certificate. The GUI will provide reason as default value but allow changing.
   * This method returns immediately.
   * If callers need to verify success of revocation they should connect
   * to certificateRevoked().
   * @param authority The subject hash value of the authority issuing the certificate to be revoked.
   * @param serial The Serial number of the certificate to be revoked.
   * @param reason Reason of revocation as specified in RevocationReasons, defaults to -1.
   */
  Q_SCRIPTABLE void revokeCertificate(const quint32 authority, const quint64 serial, const int reason = 0);

  /**
   * Signs a certification request.
   * Callers should connect to requestSigned() to be notified about any results.
   * Dates, if given, must be specified in Qt::ISODate format (YYYY-MM-DDTHH:MM:SS).
   * @param filename Certification request file to be signed
   * @param authority Subject hash value of certification authority to be used, defaults to 0.
   * @param notBefore Start of validity period, set now if empty (default).
   * @param notAfter End of validity period, set to notBefore + authority's default runtime if empty (default):
   */
  Q_SCRIPTABLE void signRequest(const QString &filename, const quint32 authority = 0,
                                const QString &notBefore = QString(), const QString &notAfter = QString());

  /**
   * Verifies validity of a certificate
   * @param authority Subject hash of authority issuing the certificate to be validated.
   * @param serial Serial of the certificate to be validated.
   * @return One value from enum States indicating result of certificate verification.
   */
  Q_SCRIPTABLE int verifyCertificate(const quint32 authority, const quint64 serial);
  /**
   * Verifies validity of a certificate
   * @param filename Filename of certificate to be validated
   * @return One value from enum States indicating result of certificate verification.
   */
  Q_SCRIPTABLE int verifyCertificate(const QString &filename);
};

}; // End namespace DBus
}; // End namespace kCA

#endif // KCADBUSINTERFACE_H
