/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "kcadbusinterface.h"
#include "kcadbustypes.h"

#include "kca.h"
#include "authority.h"
#include "database.h"

#include <libkca_ossl/commons.h>
#include <libkca_ossl/certificate.h>
#include <libkca_ossl/request.h>
#include <libkca_ossl/x509name.h>

#include <QtCore/QDateTime>
#include <QtCore/QMetaObject>
#include <QtCore/QString>
#include <QtCore/QStringList>

#include <QtDBus/QtDBus>

using namespace kCA::DBus;

Interface::Interface(QObject* parent)
    : QDBusAbstractAdaptor(parent)
{
  qDBusRegisterMetaType< kCA::DBus::Authority >();
  qDBusRegisterMetaType< QList< kCA::DBus::Authority > >();

  setAutoRelaySignals(true);

  QDBusConnection dbus = QDBusConnection::sessionBus();
  dbus.registerObject("/kCA", parent);
  dbus.registerService("de.pctiede.kCA");
}

Interface::~Interface()
{
  QDBusConnection dbus = QDBusConnection::sessionBus();
  dbus.unregisterService("de.pctiede.kCA");
  dbus.unregisterObject("/kCA", QDBusConnection::UnregisterTree);
}

QList< Authority > Interface::authorities()
{
  QList< Authority > result;
  Authority entry;
  foreach(kCA::Authority* authority, parent()->authorities.values()) {
    entry.hash = authority->hash();
    entry.name = authority->name();
    entry.subject = authority->subject();
    result.append(entry);
  }

  return result;
}

void Interface::exportCertificate(const quint32 authority, const quint64 serial, const QString& filename)
{
  if (!parent()->authorities.keys().contains(authority)) {
    emit certificateExported(authority, serial, AuthorityNotFound, "");
    return;
  }

  kCA::Database::Database *db = parent()->db_backend;
  if (!db || !db->isOpen()) {
    emit certificateExported(authority, serial, DatabaseError, "");
    return;
  }

  kCA::Database::Database::Error error;
  kCA::Database::Certificate cert = db->getCertificate(authority, serial, &error);
  if (!error) {
    kCA_ossl::Certificate(cert.certificate).toPemFile(filename);
    emit certificateExported(authority, serial, OK, filename);
    return;
  }

  emit certificateExported(authority, serial, DatabaseError, "");
}

void Interface::generateCrl(const quint32 authority, const QString& filename)
{
  if (!parent()->authorities.keys().contains(authority)) {
    emit crlGenerated(authority, AuthorityNotFound, "");
    return;
  }

  parent()->generateCrl(authority, filename);
}

void Interface::revokeCertificate(const QString& filename, const int reason)
{
  kCA_ossl::Certificate fileSrc(filename);
  if (!fileSrc.isValid()) {
    emit certificateRevoked(0, 0, FileError, reason);
    return;
  }

  revokeCertificate(fileSrc.issuerHash(), fileSrc.serial(), reason);
}

void Interface::revokeCertificate(const quint32 authority, const quint64 serial, const int reason)
{
  if (reason < 0 || reason > kCA_ossl::CertificateHold) {
    emit certificateRevoked(authority, serial, reason, ParameterError);
    return;
  }

  if (!parent()->authorities.keys().contains(authority)) {
    emit certificateExported(authority, serial, AuthorityNotFound, "");
    return;
  }

  parent()->revokeCertificate(authority, serial, reason);
}

void Interface::signRequest(const QString& filename, const quint32 authority,
                                   const QString& notBefore, const QString& notAfter)
{
  kCA_ossl::Request req(filename);
  if (!req.isValid()) {
    emit requestSigned(filename, FileError, authority, 0);
    return;
  }

  QString subject = req.subject().toString();
  if (authority && !parent()->authorities.keys().contains(authority)) {
    emit requestSigned(subject, AuthorityNotFound, authority, 0);
    return;
  }

  QDateTime startValid = QDateTime::fromString(notBefore, Qt::ISODate);
  QDateTime stopValid = QDateTime::fromString(notAfter, Qt::ISODate);
  startValid.setTimeSpec(Qt::LocalTime);
  stopValid.setTimeSpec(Qt::LocalTime);

  if (startValid.isValid() && startValid < QDateTime::currentDateTime()) {
    emit requestSigned(subject, ParameterError, authority, 0);
    return;
  }
  if (stopValid.isValid() && (stopValid < QDateTime::currentDateTime() ||
        (startValid.isValid() && stopValid < startValid))) {
    emit requestSigned(subject, ParameterError, authority, 0);
    return;
  }

  parent()->signRequest(authority, req, startValid, stopValid);
}

int Interface::verifyCertificate(const quint32 authority, const quint64 serial)
{
  if (!parent()->authorities.keys().contains(authority))
    return AuthorityNotFound;

  kCA::Database::Database *db = parent()->db_backend;
  if (!db || !db->isOpen())
    return DatabaseError;

  kCA::Database::Database::Error error;
  kCA::Database::Certificate certificate = db->getCertificate(authority, serial, &error);
  switch (error) {
    case kCA::Database::Database::NoError:
      switch (certificate.state) {
        case kCA::Database::Certificate::Valid:
          return OK;
        case kCA::Database::Certificate::Revoked:
          return CertificateRevoked;
        default:
          return CertificateExpired;
      }
    case kCA::Database::Database::NotFound:
      return SerialError;
    default:
      return DatabaseError;
  }
}

int Interface::verifyCertificate(const QString& filename)
{
  kCA_ossl::Certificate fileSrc(filename);
  if (!fileSrc.isValid())
    return FileError;

  return verifyCertificate(fileSrc.issuerHash(), fileSrc.serial());
}

#include "kcadbusinterface.moc"
