/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2007 - 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "kca.h"
#include "kcaview.h"
#include "settings.h"

#include "authority.h"
#include "database.h"
#include "kcadbusinterface.h"
#include "kcadbustypes.h"
#include "importauthorities.h"
#include "unlockauthority.h"
#include "signrequestdialog.h"

#ifdef USE_SQLITE3
#include "backends/sqlite_db.h"
#endif // USE_SQLITE3

#include <libkca_ossl/certificate.h>
#include <libkca_ossl/request.h>
#include <libkca_ossl/revocationlist.h>
#include <libkca_ossl/x509name.h>

#include <QtCore/QDateTime>
#include <QtCore/QHash>
#include <QtCore/QFile>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QThread>
#include <QtCore/QTimer>

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>
#include <QtDBus/QDBusReply>

#include <QtGui/QDropEvent>
#include <QtGui/QMessageBox>
#include <QtGui/QProgressDialog>

#include <kconfigdialog.h>
#include <kstatusbar.h>

#include <kaction.h>
#include <kactioncollection.h>
#include <kfiledialog.h>
#include <kstandardaction.h>
#include <kstandarddirs.h>

#include <KDE/KLocale>

#include <unistd.h>

using namespace kCA;

kca::kca()
    : KXmlGuiWindow(),
      db_backend(0),
      m_view(new kcaView(this)),
      dbus_interface(new DBus::Interface(this))
{
    // accept dnd
    setAcceptDrops(true);

    // tell the KXmlGuiWindow that this is indeed the main widget
    setCentralWidget(m_view);

    // then, setup our actions
    setupActions();

    // add a status bar
    statusBar()->show();

    // a call to KXmlGuiWindow::setupGUI() populates the GUI
    // with actions, using KXMLGUI.
    // It also applies the saved mainwindow settings, if any, and ask the
    // mainwindow to automatically save settings if changed: window size,
    // toolbar position, icon size, etc.
    setupGUI();

    connect(m_view->btnLock, SIGNAL(pressed()), SLOT(lockAuthority()));
    connect(m_view->pbExport, SIGNAL(pressed()), SLOT(exportCertificate()));

    connect(m_view, SIGNAL(authoritySelected(quint32)), SLOT(selectAuthority(quint32)));
    connect(m_view, SIGNAL(generateCrl()), SLOT(generateSingleCrl()));
    connect(m_view, SIGNAL(generateAllCrls()), SLOT(generateAllCrls()));

    QTimer::singleShot(0, this, SLOT(delayedInit()));
}

kca::~kca()
{
  authorities.clear();
}

QStringList kca::instanceList() const
{
  // Shamelessly stolen from kmymoney, thanks guys.
  QStringList list;
  QDBusReply<QStringList> reply = QDBusConnection::sessionBus().interface()->registeredServiceNames();

  if (reply.isValid()) {
    QStringList apps = reply.value();

    // build a list of service names of all running kmymoney applications without this one
    for (QStringList::const_iterator it = apps.constBegin(); it != apps.constEnd(); ++it) {
      // please shange this method of creating a list of 'all the other instances that are running on the system'
      // since assuming that D-Bus creates service names with -PID is an observation I don't think that it's documented somwhere
      if ((*it).indexOf("org.kde.kca-") == 0) {
        uint thisProcPid = getpid();
        if ((*it).indexOf(QString("org.kde.kca-%1").arg(thisProcPid)) != 0)
          list += (*it);
      }
    }
  }

  return list;
}

void kca::setupActions()
{
    KStandardAction::quit(qApp, SLOT(closeAllWindows()), actionCollection());

    KStandardAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

    KAction *openRequest = new KAction(KIcon("view-certificate-sign"), i18n("&Sign Request"), this);
    actionCollection()->addAction(QLatin1String("requestOpen_action"), openRequest);
    connect(openRequest, SIGNAL(triggered(bool)), SLOT(openRequest()));

    KAction *import = new KAction(KIcon("view-certificate-import"), i18n("&Import Authorities"), this);
    actionCollection()->addAction(QLatin1String("import_action"), import);
    connect(import, SIGNAL(triggered(bool)), SLOT(importAuthorities()));
}

void kca::delayedInit()
{
  attachToDatabase();
}

void kca::attachToDatabase()
{
  m_view->clearAuthorities();

  //TODO: Add code to support more database backends
  QString db_file = Settings::db_file();
  if (db_file.isEmpty())
    db_file = KStandardDirs::locateLocal("appdata", "kca.db");

  if (!QFile(db_file).exists())
    Database::SQLiteDatabase::init(db_file);

  if (!db_backend)
    db_backend = new Database::SQLiteDatabase(db_file);
  if (!db_backend) {
    QMessageBox::critical(this, i18n("Database connection"), i18n("Could not connect to database."), QMessageBox::Ok);
    return;
  }
  if (db_backend->isOpen() == 1) {
    if (QMessageBox::warning(this, i18n("Database upgrade required"),
                             i18n("Current database needs an upgrade to work with this program version. Do you want to upgrade now?"),
                             QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
      QThread *dbUpgrade = kCA::Database::SQLiteDatabase::upgrade(db_file);
      connect(dbUpgrade, SIGNAL(finished()), SLOT(attachToDatabase()));
    }
    delete db_backend;
    db_backend = 0;
    return;
  }
  else if (db_backend->isOpen()) {
    QMessageBox::critical(this, i18n("Database connection"),
                          i18n("Unable to open database.\nBackend error message is '%1'.").arg(db_backend->lastError()),
                          QMessageBox::Ok);
    return;
  }

  Authority* authority;
  foreach (ulong hash, db_backend->getAuthorities().keys()) {
    authority = new Authority(db_backend, hash, this);

    connect(authority, SIGNAL(keyLocked(quint32)), m_view, SLOT(authorityLocked(quint32)));
    connect(authority, SIGNAL(keyUnlocked(quint32)), m_view, SLOT(authorityUnlocked(quint32)));

    connect(authority, SIGNAL(requestSigned(kCA_ossl::Certificate,int)), SLOT(requestSigned(kCA_ossl::Certificate,int)));
    connect(authority, SIGNAL(certificateRevoked(quint32,quint64,int,int)), SLOT(certificateRevoked(quint32,quint64,int,int)));
    connect(authority, SIGNAL(crlGenerated(kCA_ossl::Revocationlist,int)), SLOT(crlGenerated(kCA_ossl::Revocationlist,int)));

    m_view->addAuthority(authority->issuerHash(), hash, authority->name(), authority->keyUnlocked(), authority->lockableKey());
    authorities.insert(hash, authority);
  }
}

void kca::openRequest()
{
  QString requestFile = KFileDialog::getOpenFileName(KUrl(), "application/pkcs10 text/plain", this,
                                                     i18n("Open certificate request..."));
  if (requestFile.isEmpty())
    return;

  kCA_ossl::Request req(requestFile);
  if (!req.isValid()) {
    QMessageBox::critical(this, i18n("Not a request file"), i18n("The selected file does not contain a "
                                                                 "certificate signing request."));
    return;
  }

  signRequest(m_view->activeAuthority(), req);
}

void kca::importAuthorities()
{
  QString configfile;
  QStringList authorities;
  bool importKeys;
  importAuthoritiesView *importer = new importAuthoritiesView(&configfile, &authorities, &importKeys, this);
  importer->setAttribute(Qt::WA_DeleteOnClose);
  importer->setWindowModality(Qt::ApplicationModal);
  if (importer->exec() == QDialog::Accepted) {
    int max = authorities.count();
    QProgressDialog progress(i18n("Importing authorities..."), i18n("Abort"), 0, max-1, this);
    progress.setWindowModality(Qt::ApplicationModal);
    for (int i = 0; i < max; ++i) {
      progress.setValue(i);
      Authority::importAuthority(db_backend, configfile, authorities.at(i), importKeys);
    }
    attachToDatabase();
    progress.reset();
  }
}

void kca::optionsPreferences()
{
    // The preference dialog is derived from prefs_base.ui
    //
    // compare the names of the widgets in the .ui file
    // to the names of the variables in the .kcfg file
    //avoid to have 2 dialogs shown
    if ( KConfigDialog::showDialog( "settings" ) )  {
        return;
    }
    KConfigDialog *dialog = new KConfigDialog(this, "settings", Settings::self());
    QWidget *generalSettingsDlg = new QWidget;
    ui_prefs_base.setupUi(generalSettingsDlg);
    dialog->addPage(generalSettingsDlg, i18n("General"), "package_setting");
    connect(dialog, SIGNAL(settingsChanged(QString)), m_view, SLOT(settingsChanged()));
    dialog->setAttribute( Qt::WA_DeleteOnClose );
    dialog->show();
}

void kca::selectAuthority(const quint32 authority)
{
  QList< Database::Certificate > authorityCertificates = authorities.value(authority)->certificates().values();
  QList< QStringList > details;

  QStringList detail;
  foreach(Database::Certificate certificate, authorityCertificates) {
    detail.clear();
    detail << QString().number(certificate.serial, 16).toUpper();
    detail << certificate.subject;
    detail << certificate.issued.toLocalTime().toString(Qt::ISODate);
    detail << certificate.owner;
    detail << certificate.validFrom.toLocalTime().toString(Qt::ISODate);
    detail << certificate.expires.toLocalTime().toString(Qt::ISODate);
    detail << QString().number(certificate.state, 10);
    details << detail;
  }

  m_view->showCertificates(details);
}

void kca::lockAuthority()
{
  authorities.value(m_view->activeAuthority())->lockKey();
}

bool kca::unlockAuthority(const quint32 authority, const AuthorityAction action, const QString& details)
{
  Authority* ca = authorities.value(authority, 0);
  if (!ca)
    return false;

  if (ca->keyUnlocked() || ca->unlockKey())
    return true;

  QString message;
  switch (action) {
    case GenerateCrl:
      message = i18n("Please enter file containing private key and passphrase for Authority '%1' "
                       "to generate its CRL.").arg(ca->name());
      break;
    case SignRequest:
      message = i18n("Please enter file containing private key and passphrase for Authority '%1' "
                       "to sign request with subject '%2'.").arg(ca->name()).arg(details);
      break;
    case RevokeCertificate:
      message = i18n("Please enter file containing private key and passphrase for Authority '%1' "
                       "to revoke certificate with serial '%2'.").arg(ca->name()).arg(details);
      break;
  }

  UnlockAuthority::Answer dialogAnswer;
  dialogAnswer.keyfile = ca->keyPath();

  UnlockAuthority *unlocker = new UnlockAuthority(message, &dialogAnswer, this);
  unlocker->setAttribute(Qt::WA_DeleteOnClose);
  unlocker->setWindowModality(Qt::ApplicationModal);
  if (unlocker->exec() == QDialog::Accepted) {
    return ca->unlockKey(dialogAnswer.passphrase, dialogAnswer.noAutoLock,
                         dialogAnswer.keyfile, dialogAnswer.storeKeyfile);
  }
  else
    return false;
}

void kca::signRequest(const quint32 authority, const kCA_ossl::Request& request,
                      const QDateTime& notBefore, const QDateTime& notAfter)
{
  // Check timestamps to be valid and within boundaries, if not, use authority defaults
  QDateTime localStart = QDateTime::currentDateTime();
  if (notBefore.isValid() && notBefore >= localStart)
    localStart = notBefore;
  QDateTime localStop = QDateTime::currentDateTime().addDays(authorities.value(authority)->certificateDays());
  if (notAfter.isValid() && notAfter > notBefore)
    localStop = notBefore;

  // Set defaults for signing, modifiable by dailog
  SignRequestDialog::Settings signingSettings(authority, Settings::supersede(), localStart, localStop);

  SignRequestDialog *dlg = new SignRequestDialog(authorities, request, &signingSettings, this);
  dlg->setAttribute(Qt::WA_DeleteOnClose);
  if (dlg->exec() == QDialog::Accepted) {
    if (unlockAuthority(signingSettings.authority, SignRequest, request.subject().toString())) {
      authorities.value(signingSettings.authority)->signRequest("", request,
                                                                signingSettings.notBefore, signingSettings.notAfter,
                                                                signingSettings.extensions, signingSettings.supersede);
    }
    else
      requestSigned(kCA_ossl::Certificate(), Authority::KeyNotLoaded);
  }
}

void kca::revokeCertificate(const quint32 authority, const quint64 serial, const int reason)
{
  kCA::Authority *ca = authorities.value(authority, 0);
  if (!ca)
    return;

  if (!ca->certificates().keys().contains(serial)) {
    certificateRevoked(authority, serial, reason, DBus::SerialError);
    return;
  }

  if (unlockAuthority(authority, RevokeCertificate, QString().number(serial, 16).toUpper())) {
    ca->revokeCertificate(serial,(kCA_ossl::RevocationReason) reason, QDateTime());
  }
  else
    certificateRevoked(authority, serial, reason, DBus::AuthorityLocked);
}

void kca::generateCrl(const quint32 authority, const QString& targetfile)
{
  // authority == 0 means CRLs for all authorities should be produced.
  if (authority == 0) {
    foreach(quint32 key, authorities.keys())
      generateCrl(key, targetfile);
    // Exit recursion!
    return;
  }

  if (!targetfile.isEmpty())
    crlExports.insert(authority, targetfile);

  kCA::Authority *ca = authorities.value(authority, 0);
  if (!ca)
    return;

  if (unlockAuthority(authority, GenerateCrl)) {
    ca->generateCrl();
  }
  else
    emit dbus_interface->crlGenerated(authority, DBus::AuthorityLocked, "");
}

void kca::generateSingleCrl()
{
  generateCrl(m_view->activeAuthority(), "");
}

void kca::generateAllCrls()
{
  generateCrl(0, "");
}

void kca::exportCertificate()
{
  bool result;
  quint32 authority = m_view->activeAuthority();
  quint64 certificateSerial = m_view->activeCertificate();
  kCA_ossl::Certificate cert(authorities.value(authority)->certificates().value(certificateSerial).certificate);

  QString targetFileName = KFileDialog::getSaveFileName(KUrl(QString().number(certificateSerial, 16).toUpper()),
                                                        "application/x-x509-ca-cert", this,
                                                        i18n("Save certificate as..."));
  if (targetFileName.isEmpty())
    return;

  if (targetFileName.toLower().endsWith(".der"))
    result = cert.toDerFile(targetFileName);
  else
    result = cert.toPemFile(targetFileName);

  emit dbus_interface->certificateExported(authority, certificateSerial,
                                           (result ? DBus::OK : DBus::FileError), targetFileName);
}

void kca::requestSigned(const kCA_ossl::Certificate& certificate, const int error)
{
  if (!error && m_view->activeAuthority() == certificate.issuerHash())
    m_view->addCertificate(QStringList()
                              << QString().number(certificate.serial(), 16).toUpper()
                              << certificate.subject().toString()
                              << QDateTime::currentDateTime().toLocalTime().toString(Qt::ISODate)
                              << ""
                              << certificate.notBefore().toLocalTime().toString(Qt::ISODate)
                              << certificate.notAfter().toLocalTime().toString(Qt::ISODate));

  emit dbus_interface->requestSigned(certificate.subject().toString(), error,
                                     certificate.issuerHash(), certificate.serial());

  QString errorText = i18n("Request signing failed for unknown reason.");
  switch (error) {
    case Authority::NoError:
      return;
    case Authority::KeyNotLoaded:
      errorText = i18n("Authority's private key necessary for request signing was not loaded.");
      break;
    case Authority::DatabaseError:
      errorText = i18n("Database error. Backend error message was:\n'%1'").arg(db_backend->lastError());
      break;
    case Authority::ConstraintError:
      errorText = i18n("Constraints of parameters were not matched.");
      break;
    case Authority::TimeoutError:
      errorText = i18n("Searching a free serial number timed out.");
      break;
    case Authority::RandomizerError:
      errorText = i18n("Error in random number generator during serial number generation.");
      break;
    case Authority::SigningError:
      errorText = i18n("Backend library libkca_ossl failed to sign request and generate certificate.");
      break;
  }
  QMessageBox::warning(this, i18n("Request signing"), errorText, QMessageBox::Ok);
}

void kca::certificateRevoked(const quint32 authority, const quint64 serial, const int reason, const int error)
{
  emit dbus_interface->certificateRevoked(authority, serial, reason, error);
}

void kca::crlGenerated(const kCA_ossl::Revocationlist& crl, const int error)
{
  if (error) {
    // crl is empty, so try to get issuer subject hash from sender()
    Authority *ca =(Authority*) sender();
    if (ca) {
      emit dbus_interface->crlGenerated(ca->hash(), DBus::DatabaseError, "");

      QString errorText = i18n("CRL generation failed for unknown reason.");
      switch (error) {
        case Authority::KeyNotLoaded:
          errorText = i18n("Authority's private key necessary for CRL signing was not loaded.");
          break;
        case Authority::DatabaseError:
          errorText = i18n("Database error. Backend error message was:\n'%1'").arg(db_backend->lastError());
          break;
        case Authority::SigningError:
          errorText = i18n("Backend library libkca_ossl failed to sign CRL.");
          break;
      }
      QMessageBox::warning(this, i18n("Revocationlist generation"), errorText, QMessageBox::Ok);
      //TODO: Add code to log database error message.
    }
    return;
  }

  quint32 crlIssuer = crl.issuerHash();
  QStringList files = crlExports.values(crlIssuer);
  foreach(QString target, files) {
    emit dbus_interface->crlGenerated(crlIssuer,
                                      (crl.toPemFile(target) ? DBus::OK : DBus::FileError), target);
    crlExports.remove(crlIssuer, target);
  }
}

#include "kca.moc"
