/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef DATABASE_H
#define DATABASE_H

#include <libkca_ossl/commons.h>

#include <QtCore/QtGlobal>
#include <QtCore/QByteArray>
#include <QtCore/QDateTime>
#include <QtCore/QList>
#include <QtCore/QString>

template<typename Key, typename T> class QHash;

namespace kCA
{
namespace Database
{

/** Datatype to request/store a certificate. */
struct Certificate {
  /** Allowed states of a certificate. */
  enum State {
    Valid,    /**<Certificate is valid. */
    Expired,  /**<Certificate has expired. */
    Revoked,  /**<Certificate has been revoked. */
  };

  /** Certificate's serial number. */
  quint64 serial;
  /** Certificate's SHA1 fingerprint. */
  QByteArray fingerprint;
  /** Certificate's issuer hash. */
  quint32 authorityhash;

  /** Certificate owner. */
  QString owner;

  /** Certificate's subject. */
  QString subject;

  /** Certificate issue date. */
  QDateTime issued;
  /** Certificate validity start date. */
  QDateTime validFrom;
  /** Certificate expiration date. */
  QDateTime expires;

  /** Certificate state. */
  State state;

  /** Certificate revocation date. */
  QDateTime revoked;
  /** Certificate revocation reason. */
  kCA_ossl::RevocationReason reason;

  /** Certificate raw data. */
  QByteArray certificate;
};

/** Datatype to request/store extensions. */
struct Extension {
  /** Allowed types of extensions. */
  enum ExtensionType {
    Certificate,  /**<Extension to use for certificates issued by authority. */
    CRL,          /**<Extension to use for certificate revocation lists issued by authority. */
  };

  /** Extension ID from database. */
  int id;
  /** Extension OID. */
  QString oid;
  /** Extension value. */
  QByteArray value;
  /** State of criticality. */
  bool critical;
  /** Type of extension. */
  ExtensionType type;
};

/** Datatype to request/store policies. */
struct Policy {
  /** Allowed values for data field policy. */
  enum FieldSetting {
    Supplied, /**<Data field must be supplied to match policy. */
    Match,    /**<Field value must match authorities field value. */
    Optional  /**<Optional value. */
  };

  /** Make a given string a value from FieldSetting. */
  static inline FieldSetting convert(const QString& value)
  {
    if (value.toLower() == "supplied")
      return Supplied;
    else if (value.toLower() == "match")
      return Match;
    else // if (value.toLower() == "optional")
      return Optional;
  }

  /** Name of policy for identification. */
  QString name;

  /** Country policy, defaults to Supplied. */
  FieldSetting country;
  /** State policy, defaults to Optional. */
  FieldSetting state;
  /** Location policy, defaults to Supplied. */
  FieldSetting location;
  /** Organization polcy, defaults to Supplied. */
  FieldSetting organization;
  /** Organizational Unit policy, defaults to Optional. */
  FieldSetting organizationalUnit;
  /** Common Name policy, defaults to Supplied. */
  FieldSetting commonName;
  /** Email policy, defaults to Supplied. */
  FieldSetting email;
};

/** Datatype to request/store a certification authority. */
struct Authority {
  /** Name of authority. */
  QString name;
  /** Hash value of authority. */
  quint32 hash;

  /** Authority's certificate raw data. */
  QByteArray certificate;
  /** Authority's private key raw data. */
  QByteArray key;
  /** Authority's private key has changed. */
  bool keyDirty;

  /** Digest algorithm used at every signing operation. */
  kCA_ossl::Digest digest;

  /** Current CRL serial number. */
  unsigned long crlnumber;
  /** Default validity time of issued CRL in days. */
  unsigned int crldays;

  /** Default validity time of issued certificates in days. */
  unsigned int certificateDays;

  /** Policy for certificate requests to match. */
  Policy policy;
  /** Extensions associated with certification authority. */
  QList< Extension > extensions;
};

/**
 * Interface definition for database backends.
 * At any method returning a pointer or a QList of pointers
 * the consumer is responsible for deletion of objects.
 *
 * @author Felix Tiede <info@pc-tiede.de>
 */
class Database
{
  public:
    /** Errors as reported from database backend. */
    enum Error {
      NoError,      /**<No error in operation. */
      NotOpen,      /**<Database is not open. */
      NotFound,     /**<Requested data not found by given key. */
      RequestError, /**<Illegal request or constraint not matched. */
      TransactionError, /**<Unable to write changes due to an error during transaction. */
      ConstraintError,  /**<Constraints not matched for opertaiton. */
      PreparationError, /**<Database interface was not prepared correctly. */
    };

    /** Default destructor. */
    virtual ~Database() {}

    /**
     * Initialize database.
     * Override by database interface implementation.
     */
    static bool init() {return false;}

    /**
     * Upgrade database structure.
     * Override by database interface implementation.
     */
    static bool upgrade() {return false;}

    /**
     * Check if connection is open and database is able to handle requests.
     * @return 0, if database is open, 1, if an upgrade is required and &gt;1 if another error occurred.
     */
    virtual int isOpen() const = 0;

    /**
     * Get last error message from database subsystem.
     * The message is backend specific.
     * @return Last error message from database backend.
     */
    virtual QString lastError() const = 0;

    /**
     * Get all available authorities.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of certification authorities. Empty list if an error occurred.
     */
    virtual const QHash< quint32, Authority > getAuthorities(Error* error = 0) const = 0;
    /**
     * Get certification authority based on its hash value.
     * @param[in] authorityhash Hash value of authority to load.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certification authority designated by name.
     */
    virtual Authority getAuthority(const quint32 authorityhash, Error* error = 0) const = 0;
    /**
     * Get certification authority based on its friendly name.
     * @param[in] name Friendly name of authority to load.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certification authority designated by name.
     */
    virtual Authority getAuthority(const QString& name, Error* error = 0) const = 0;
    /**
     * Set certification authority. If hash value matches an existing
     * certification authority, that authority is overwritten, so use with care.
     * @param[in] authority Certification authority to store in database.
     * @return Result of database operation.
     */
    virtual Error setAuthority(const Authority& authority) = 0;
    /**
     * Update certification suthority certificate and private key.
     * For private key either an empty string, a pathname or PEM-encoded raw data is accepted.
     * If key is empty, authority's private key is left unchanged. use with care, as zhid
     * qill cause serious trouble if current private key is not a pathname.
     * @param[in] authorityhash Certification authority to update certificate and key.
     * @param[in] certificate PEM-encoded certificate raw data to set as authority certificate.
     * @param[in] key Pathname to private key or PEM-encoded raw private key data, defaults to empty.
     * @return Result of database operation.
     */
    virtual Error setAuthorityCertificate(const quint32 authorityhash,
                                          const QByteArray& certificate,
                                          const QByteArray& key = QByteArray()) = 0;

    /**
     * Get a list of available policies for certification authorities.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of policies or empty list if an error occurred.
     */
    virtual const QHash< QString, Policy > getPolicies(Error* error = 0) const = 0;
    /**
     * Get a list of available extensions for certification authorities,
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of extensions or empty list if an error occurred.
     */
    virtual const QList< Extension > getExtensions(Error* error = 0) const = 0;

    /**
     * Get a list of available certificates for certification authority if given.
     * @param[in] authorityhash Hash value of authority of which certificates should be loaded. If 0, all certificates are loaded.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of certificates or empty list if an error occurred.
     */
    virtual const QHash< quint64, Certificate > getCertificates(const quint32 authorityhash = 0,
                                                                Error* error = 0) const = 0;
    /**
     * Get certificate based on its SHA-1 fingerprint value.
     * @param[in] fingerprint Hex-encoded fingerprint value of certificate to load.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certificate designated by hash if existent.
     */
    virtual Certificate getCertificate(const QByteArray& fingerprint, Error* error = 0) const = 0;
    /**
     * Get certificate based on its issuer hash value and serial.
     * @param[in] authorityhash Hash value of issuing certification authority.
     * @param[in] serial Serial number of certificate.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certificate designated by hash if existent.
     */
    virtual Certificate getCertificate(const quint32 authorityhash, const quint64 serial, Error* error = 0) const = 0;
    /**
     * Set certificate. If hash value matches an existing certificate,
     * that certificate is overwritten, so use with care.
     * @param[in] certificate Certificate to store in database.
     * @return Result of database operation.
     */
    virtual Error setCertificate(const Certificate& certificate) = 0;

    /**
     * Find out if a serial is already in use by another certificate from the same certification authority.
     * @param[in] authorityhash Hash value of authority to check serial against.
     * @param[in] serial Serial number to be found.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return False, if serial is not in use by the given certification authority, true otherwise (also, if an error occurred.
     */
    virtual bool isSerialUsed(const quint32 authorityhash, const quint64 serial, Error* error = 0) const = 0;
};

}; // End namespace Database
}; // End namespace kCA

#endif // DATABASE_H
