/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010, 2011 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef SQLITEDATABASE_H
#define SQLITEDATABASE_H

#include "kca_backends_export.h"

#include "../database.h"

#include <string>

class QString;
class QThread;
template<typename key, typename T> class QHash;
template<typename T> class QList;

namespace kCA
{
namespace Database
{

class SQLiteDatabasePrivate;

/**
 * SQLite3 database backend implementation. May also serve as reference
 * implementation for other database backends.
 * @short SQLite3 database backend
 * @author Felix Tiede <info@pc-tiede.de>
 */
class _LIBKCABACKENDS_EXPORT SQLiteDatabase : public Database
{
  friend class SQLiteDatabasePrivate;

  public:
    /** Open SQLite3 database at filename for operations. */
    SQLiteDatabase(const QString& filename);
    ~SQLiteDatabase();

    /**
     * Initialize database.
     * This operation destroys any data in an existing
     * database file.
     */
    static bool init(const QString& filename);

    /**
     * Upgrade database structure.
     * This operation updates the database with any structural changes.
     */
    static QThread* upgrade(const QString& filename);

    /**
     * Check if connection is open and database is able to handle requests.
     * @return 0, if database is open, 1, if an upgrade is required and &gt;1 if another error occurred.
     */
    int isOpen() const;

    /**
     * Get last error message from database subsystem.
     * The message is backend specific.
     * @return Last error message from database backend.
     */
    QString lastError() const;

    /**
     * Get all available authorities.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of certification authorities. Empty list if an error occurred.
     */
    const QHash< quint32, Authority > getAuthorities(Error* error = 0) const;
    /**
     * Get certification authority based on its hash value.
     * @param[in] authorityhash Hash value of authority to load.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certification authority designated by name.
     */
    Authority getAuthority(const quint32 authorityhash, Error* error = 0) const;
    /**
     * Get certification authority based on its friendly name.
     * @param[in] name Friendly name of authority to load.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certification authority designated by name.
     */
    Authority getAuthority(const QString& name, Error* error = 0) const;
    /**
     * Set certification authority. If hash value matches an existing
     * certification authority, that authority is overwritten, so use with care.
     * @param[in] authority Certification authority to store in database.
     * @return Result of database operation.
     */
    Error setAuthority(const Authority& authority);
    /**
     * Update certification suthority certificate and private key.
     * For private key either an empty string, a pathname or PEM-encoded raw data is accepted.
     * If @param key is empty, authority's private key is left unchanged. use with care, as zhid
     * qill cause serious trouble if current private key is not a pathname.
     * @param[in] authorityhash Certification authority to update certificate and key.
     * @param[in] certificate PEM-encoded certificate raw data to set as authority certificate.
     * @param[in] key Pathname to private key or PEM-encoded raw private key data, defaults to empty.
     * @return Result of database operation.
     */
    Error setAuthorityCertificate(const quint32 authorityhash,
                                  const QByteArray& certificate,
                                  const QByteArray& key = QByteArray());

    /**
     * Get a list of available policies for certification authorities.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of policies or empty list if an error occurred.
     */
    const QHash< QString, Policy > getPolicies(Error* error = 0) const;
    /**
     * Get a list of available extensions for certification authorities,
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of extensions or empty list if an error occurred.
     */
    const QList< Extension > getExtensions(Error* error = 0) const;

    /**
     * Get a list of available certificates for certification authority if given.
     * @param[in] authorityhash Hash value of authority of which certificates should be loaded. If 0, all certificates are loaded.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return List of certificates or empty list if an error occurred.
     */
    const QHash< quint64, Certificate > getCertificates(const quint32 authorityhash = 0, Error* error = 0) const;
    /**
     * Get certificate based on its SHA-1 fingerprint value.
     * @param[in] fingerprint Hex-encoded fingerprint value of certificate to load.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certificate designated by hash if existent.
     */
    Certificate getCertificate(const QByteArray& fingerprint, Error* error = 0) const;
    /**
     * Get certificate based on its issuer hash value and serial.
     * @param[in] authorityhash Hash value of issuing certification authority.
     * @param[in] serial Serial number of certificate.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return Certificate designated by hash if existent.
     */
    Certificate getCertificate(const quint32 authorityhash, const quint64 serial, Error* error = 0) const;
    /**
     * Set certificate. If hash value matches an existing certificate,
     * that certificate is overwritten, so use with care.
     * @param[in] certificate Certificate to store in database.
     * @return Result of database operation.
     */
    Error setCertificate(const Certificate& certificate);

    /**
     * Find out if a serial is already in use by another certificate from the same certification authority.
     * @param[in] authorityhash Hash value of authority to check serial against.
     * @param[in] serial Serial number to be found.
     * @param[out] error If non-null it will carry an error code from the database.
     * @return False, if serial is not in use by the given certification authority, true otherwise (also, if an error occurred.
     */
    bool isSerialUsed(const quint32 authorityhash, const quint64 serial, Error* error = 0) const;

  private:
    /** Table specifications for database initialization. */
    static const std::string tablespecs[];
    /** Upgrade specifications for database. */
    static const std::string upgradespecs[];

    /** Carrier for prepared SQLite statements. */
    SQLiteDatabasePrivate * const d;
};

}; // End namespace Database
}; // End namespace kCA

#endif // SQLITEDATABASE_H
