/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010, 2011 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "sqlite_db.h"
#include "../database.h"

#include "sqlite_db_upgrade.h"

#define _DB_VERSION "2"

#include <libkca_ossl/commons.h>

#include <QtCore/QFile>
#include <QtCore/QHash>
#include <QtCore/QList>
#include <QtCore/QObject>
#include <QtCore/QRegExp>
#include <QtCore/QString>
#include <QtCore/QThread>

#include <sqlite3.h>

namespace kCA
{
namespace Database
{
class SQLiteDatabasePrivate
{
public:
  SQLiteDatabasePrivate(const QString& filename) :
    upgradeRequired(false),
    handle(0),
    authorities(0),
    authority_policy(0),
    authority_extensions(0),
    extensions(0),
    policies(0),
    authority_serial(0),
    certificate(0)
  {
    QFile db(filename);

    if (!db.exists())
      return;

    int db_flags = SQLITE_OPEN_READWRITE | SQLITE_OPEN_FULLMUTEX | SQLITE_OPEN_PRIVATECACHE;
    sqlite3_open_v2(filename.toUtf8().constData(), &handle, db_flags, NULL);
    sqlite3_exec(handle, "PRAGMA foreign_keys = ON;", NULL, NULL, NULL);

    // Figure out database version
    sqlite3_stmt *version;
    if (sqlite3_prepare_v2(handle, "SELECT 1 FROM Settings WHERE name = 'version' AND value = '" _DB_VERSION "';",
                           -1, &version, NULL) || sqlite3_step(version) != SQLITE_ROW) {
      sqlite3_finalize(version);
      sqlite3_close(handle);
      handle = 0;
      upgradeRequired = true;
      error = QObject::tr("Database version mismatch, please upgrade database structure.");
      return;
    }
    sqlite3_finalize(version);

    static const QByteArray caSql = "SELECT * FROM Authorities;";
    static const QByteArray policySql = "SELECT * FROM Policies WHERE id = ?;";
    static const QByteArray extensionsSql = "SELECT * FROM Extensions "
                                              "LEFT JOIN CAExtensions "
                                                "ON Extensions.id = CAExtensions.extension "
                                              "WHERE CAExtensions.authority = ?;";
    static const QByteArray certSql = "SELECT * FROM Certificates "
                                        "WHERE (authority = ? AND serial = ?);";

    // Prepare SQL statements for future execution
    sqlite3_prepare_v2(handle, caSql.constData(), caSql.length(), &authorities, NULL);
    sqlite3_prepare_v2(handle, policySql.constData(), policySql.length(), &authority_policy, NULL);
    sqlite3_prepare_v2(handle, extensionsSql.constData(), extensionsSql.length(), &authority_extensions, NULL);
    sqlite3_prepare_v2(handle, certSql.constData(), certSql.length(), &certificate, NULL);
  }

  ~SQLiteDatabasePrivate()
  {
    if (handle) {
      if (authorities)
        sqlite3_finalize(authorities);
      if (authority_policy)
        sqlite3_finalize(authority_policy);
      if (authority_extensions)
        sqlite3_finalize(authority_extensions);
      if (extensions)
        sqlite3_finalize(extensions);
      if (policies)
        sqlite3_finalize(policies);
      if (authority_serial)
        sqlite3_finalize(authority_serial);
      if (certificate)
        sqlite3_finalize(certificate);

      sqlite3_close(handle);
      handle = 0;
    }
  }

  /** Set to true, if database needs an upgrade. */
  bool upgradeRequired;

  /** Handle to SQLite database. */
  sqlite3* handle;

  /** Prepared statements for recurring queries. */
  sqlite3_stmt *authorities, *authority_policy, *authority_extensions;
  sqlite3_stmt *extensions, *policies;
  sqlite3_stmt *authority_serial;
  sqlite3_stmt *certificate;

  /** Last error message from SQLite backend. */
  mutable QString error;

  /** Convert a SQL result to Authority. */
  static inline Authority toAuthority(sqlite3_stmt* statement)
  {
    Authority result;
    result.keyDirty = false;
    result.hash =(quint32) sqlite3_column_int64(statement, 1);
    result.name = QString((const char*) sqlite3_column_text(statement, 2));
    result.digest =(kCA_ossl::Digest) sqlite3_column_int(statement, 3);
    result.certificateDays = sqlite3_column_int(statement, 4);
    result.crlnumber = sqlite3_column_int(statement, 6);
    result.crldays = sqlite3_column_int(statement, 7);
    result.certificate = QByteArray((const char*) sqlite3_column_blob(statement, 8),
                                    sqlite3_column_bytes(statement, 8));
    result.key = QByteArray((const char*) sqlite3_column_blob(statement, 9),
                            sqlite3_column_bytes(statement, 9));

    return result;
  }

  /** Convert a SQL result to Extension. */
  static inline Extension toExtension(sqlite3_stmt* statement)
  {
    Extension result;
    result.id = sqlite3_column_int(statement, 0);
    result.critical = (sqlite3_column_int(statement, 2) != 0);
    result.oid = QString((const char*) sqlite3_column_text(statement, 3));
    result.value = QByteArray((const char*) sqlite3_column_text(statement, 4), sqlite3_column_bytes(statement, 4));
    result.type =(Extension::ExtensionType) sqlite3_column_int(statement, 1);

    return result;
  }

  /** Convert a SQL result to Policy. */
  static inline Policy toPolicy(sqlite3_stmt* statement)
  {
    Policy result;
    result.name = QString((const char*) sqlite3_column_text(statement, 1));
    result.country =(Policy::FieldSetting) sqlite3_column_int(statement, 2);
    result.state =(Policy::FieldSetting) sqlite3_column_int(statement, 3);
    result.location =(Policy::FieldSetting) sqlite3_column_int(statement, 4);
    result.organization =(Policy::FieldSetting) sqlite3_column_int(statement, 5);
    result.organizationalUnit =(Policy::FieldSetting) sqlite3_column_int(statement, 6);
    result.commonName =(Policy::FieldSetting) sqlite3_column_int(statement, 7);
    result.email =(Policy::FieldSetting) sqlite3_column_int(statement, 8);

    return result;
  }

  /** Convert a SQL result to Certificate. */
  static inline Certificate toCertificate(sqlite3_stmt* statement)
  {
    Certificate result;
    result.serial =(quint64) sqlite3_column_int64(statement, 1);
    result.fingerprint = QByteArray((const char*) sqlite3_column_text(statement, 2),
                                    sqlite3_column_bytes(statement, 2));
    result.authorityhash=(quint32) sqlite3_column_int64(statement, 3);
    result.owner = QString((const char*) sqlite3_column_text(statement, 8));
    result.subject = QString((const char*) sqlite3_column_text(statement, 4));
    result.issued = QDateTime::fromString(QString((const char*) sqlite3_column_text(statement, 5)), Qt::ISODate);
    result.issued.setTimeSpec(Qt::UTC);
    result.validFrom = QDateTime::fromString(QString((const char*) sqlite3_column_text(statement, 6)), Qt::ISODate);
    result.validFrom.setTimeSpec(Qt::UTC);
    result.expires = QDateTime::fromString(QString((const char*) sqlite3_column_text(statement, 7)), Qt::ISODate);
    result.expires.setTimeSpec(Qt::UTC);
    if ((result.state =(Certificate::State) sqlite3_column_int(statement, 9)) == Certificate::Revoked) {
      result.reason =(kCA_ossl::RevocationReason) sqlite3_column_int(statement, 11);
      result.revoked = QDateTime::fromString(QString((const char*) sqlite3_column_text(statement, 10)), Qt::ISODate);
      result.revoked.setTimeSpec(Qt::UTC);
    }
    else
      result.revoked = QDateTime();

    result.certificate = QByteArray((const char*) sqlite3_column_blob(statement, 12),
                                    sqlite3_column_bytes(statement, 12));

    return result;
  }
};
}; // End namespace Database
}; // End namespace kCA

using namespace kCA::Database;

const std::string SQLiteDatabase::tablespecs[] = {
  std::string("CREATE TABLE Settings (name TEXT, value TEXT, UNIQUE (name));"),
  std::string("INSERT INTO Settings (name, value) VALUES ('version', '" _DB_VERSION "');"),
  std::string("CREATE TABLE Policies (id INTEGER PRIMARY KEY AUTOINCREMENT,"
                "name TEXT UNIQUE,"
                "Country INTEGER, STate INTEGER, Location INTEGER,"
                "Organization INTEGER, OrganizationalUnit INTEGER,"
                "CommonName INTEGER, Email INTEGER,"
                "UNIQUE (Country, STate, Location, Organization, OrganizationalUnit, CommonName, Email));"),
  std::string("CREATE INDEX policy_name_idx ON Policies (name ASC);"),
  std::string("INSERT INTO Policies (name, Country, STate, Location, Organization, OrganizationalUnit, CommonName, Email)"
                "VALUES ('default', 0, 2, 0, 0, 2, 0, 0);"),

  std::string("CREATE TABLE Extensions (id INTEGER PRIMARY KEY AUTOINCREMENT,"
                "type INTEGER, critical INTEGER,"
                "oid TEXT, value TEXT, UNIQUE (type, oid, value));"),
  std::string("CREATE INDEX extension_oid_idx ON Extensions (oid ASC);"),
  std::string("CREATE INDEX extension_idx ON Extensions (type, oid ASC, value);"),

  std::string("CREATE TABLE Authorities (id INTEGER PRIMARY KEY AUTOINCREMENT,"
                "hash INTEGER UNIQUE, name TEXT, digest INTEGER, certificatedays INTEGER,"
                "policy INTEGER REFERENCES Policies (id) ON UPDATE CASCADE ON DELETE RESTRICT,"
                "crlnumber UNSIGNED BIG INT, crldays INTEGER, certificate BLOB, key BLOB);"),
  std::string("CREATE INDEX authority_name_idx ON Authorities (name ASC);"),
  std::string("CREATE INDEX authority_hash_idx ON Authorities (hash);"),

  std::string("CREATE TABLE CAExtensions (id INTEGER PRIMARY KEY AUTOINCREMENT,"
                "authority INTEGER REFERENCES Authorities (id) ON UPDATE CASCADE ON DELETE CASCADE,"
                "extension INTEGER REFERENCES Extensions (id) ON UPDATE CASCADE ON DELETE RESTRICT,"
                "UNIQUE (authority, extension));"),
  std::string("CREATE INDEX authority_extension_idx ON CAExtensions (authority);"),
  std::string("CREATE INDEX extension_authority_idx ON CAExtensions (extension);"),

  std::string("CREATE TABLE Certificates (id INTEGER PRIMARY KEY AUTOINCREMENT,"
                "serial UNSIGNED BIG INT, fingerprint TEXT UNIQUE,"
                "authority INTEGER REFERENCES Authorities (hash) ON UPDATE CASCADE ON DELETE RESTRICT,"
                "subject TEXT, issued TEXT, validfrom TEXT, expiration TEXT, owner TEXT,"
                "status INTEGER, revocation TEXT, reason INTEGER, certificate BLOB,"
                "UNIQUE (authority, serial));"),
  std::string("CREATE INDEX certificate_expiration_idx ON Certificates (status ASC, expiration ASC);"),
  std::string("CREATE INDEX certificate_authority_idx ON Certificates (authority);"),
  std::string("CREATE INDEX certificate_authority_serial_idx ON Certificates (authority, serial);"),
  std::string("CREATE INDEX certificate_status_idx ON Certificates (status, id);"),

  std::string("CREATE TRIGGER upd_expire_certificates AFTER UPDATE ON Certificates "
                "BEGIN "
                  "UPDATE Certificates SET status = 1 "
                    "WHERE (status = 0 OR (status = 2 AND reason = %CertificateHold%)) "
                      "AND expiration < strftime('%Y-%m-%dT%H:%M:%S', 'now'); "
                "END;").replace(155, 17, QString("%1").arg(kCA_ossl::CertificateHold).toUtf8().constData()),
  std::string("CREATE TRIGGER ins_expire_certificates AFTER INSERT ON Certificates "
                "BEGIN "
                  "UPDATE Certificates SET status = 1 "
                    "WHERE (status = 0 OR (status = 2 AND reason = %CertificateHold%)) "
                      "AND expiration < strftime('%Y-%m-%dT%H:%M:%S', 'now'); "
                "END;").replace(155, 17, QString("%1").arg(kCA_ossl::CertificateHold).toUtf8().constData()),
};

SQLiteDatabase::SQLiteDatabase(const QString& filename) :
    d(new SQLiteDatabasePrivate(filename))
{}

SQLiteDatabase::~SQLiteDatabase()
{
  delete d;
}

bool SQLiteDatabase::init(const QString& filename)
{
  QFile db(filename);

  if (db.exists()) {
    // Remove the old DB in order to create a new one.
    if (!db.remove())
      return false;
  }

  // Set up some flags for db connection
  int db_flags = SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE |
                  SQLITE_OPEN_FULLMUTEX | SQLITE_OPEN_PRIVATECACHE;

  sqlite3 *hnd;
  if (sqlite3_open_v2(filename.toUtf8().constData(), &hnd,
      db_flags, NULL) != SQLITE_OK || !hnd) {
    // SQLite failed to open the database file
    return false;
  }

  for (int i=0; i<21; ++i)
    sqlite3_exec(hnd, tablespecs[i].c_str(), NULL, NULL, NULL);

  sqlite3_close(hnd);
  return true;
}

QThread* SQLiteDatabase::upgrade(const QString& filename)
{
  SQLiteDatabaseUpgrade *upgrader = new SQLiteDatabaseUpgrade(filename, _DB_VERSION);

  if (upgrader->upgradeRequired() && upgrader->prepareUpgrade())
    upgrader->start();

  return upgrader;
}

int SQLiteDatabase::isOpen() const
{
  if (d->handle)
    return 0;
  else if (d->upgradeRequired)
    return 1;

  return 2;
}

QString SQLiteDatabase::lastError() const
{
  return d->error;
}

const QHash< quint32, Authority > SQLiteDatabase::getAuthorities(Database::Error* error) const
{
  QHash< quint32, Authority > result;

  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }
  if (error)
    *error = Database::NoError;

  int caResult, extResult;
  Database::Error err = Database::NoError;
  Authority authority;

  // Fetch all authorities and related subrecords.
  while ((caResult = sqlite3_step(d->authorities)) == SQLITE_ROW) {
    authority = d->toAuthority(d->authorities);

    sqlite3_bind_int(d->authority_policy, 1, sqlite3_column_int(d->authorities, 5));
    sqlite3_bind_int(d->authority_extensions, 1, sqlite3_column_int(d->authorities, 0));

    if (sqlite3_step(d->authority_policy) == SQLITE_ROW)
      authority.policy = d->toPolicy(d->authority_policy);
    else
      err = Database::RequestError;

    sqlite3_reset(d->authority_policy);

    // Fetch all extensions for authority.
    while ((extResult = sqlite3_step(d->authority_extensions)) == SQLITE_ROW)
      authority.extensions << d->toExtension(d->authority_extensions);
    sqlite3_reset(d->authority_extensions);
    if (extResult != SQLITE_DONE)
      err = Database::RequestError;

    result.insert(authority.hash, authority);
  }

  sqlite3_reset(d->authorities);

  if (caResult != SQLITE_DONE)
    err = Database::RequestError;

  if (error)
    *error = err;

  return result;
}

Authority SQLiteDatabase::getAuthority(const quint32 authorityhash, Database::Error* error) const
{
  Authority result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }

  Database::Error err = Database::NoError;

  QByteArray sql = QString("SELECT * FROM Authorities WHERE hash=%1;").arg(authorityhash).toUtf8();
  sqlite3_stmt *authority;
  if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(), &authority, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    if (error)
      *error = Database::RequestError;
    return result;
  }

  int extResult, request = sqlite3_step(authority);
  if (request == SQLITE_ROW) {
    result = d->toAuthority(authority);
    sqlite3_bind_int(d->authority_policy, 1, sqlite3_column_int(authority, 5));
    sqlite3_bind_int(d->authority_extensions, 1, sqlite3_column_int(authority, 0));

    if (sqlite3_step(d->authority_policy) == SQLITE_ROW)
      result.policy = d->toPolicy(d->authority_policy);
    else
      err = Database::RequestError;

    sqlite3_reset(d->authority_policy);

    // Fetch all extensions for authority.
    while ((extResult = sqlite3_step(d->authority_extensions)) == SQLITE_ROW)
      result.extensions << d->toExtension(d->authority_extensions);
    sqlite3_reset(d->authority_extensions);
    if (extResult != SQLITE_DONE)
      err = Database::RequestError;
  }
  else if (request == SQLITE_DONE)
    err = Database::NotFound;
  else {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    err = Database::RequestError;
  }

  sqlite3_finalize(authority);

  if (error)
    *error = err;

  return result;
}

Authority SQLiteDatabase::getAuthority(const QString& name, Database::Error* error) const
{
  Authority result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }

  Database::Error err = Database::NoError;

  QByteArray sql = QString("SELECT * FROM Authorities WHERE LOWER(name)='%1';").arg(name.toLower()).toUtf8();
  sqlite3_stmt *authority;
  if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(), &authority, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    if (error)
      *error = Database::RequestError;
    return result;
  }

  int extResult, request = sqlite3_step(authority);
  if (request == SQLITE_ROW) {
    result = d->toAuthority(authority);
    sqlite3_bind_int(d->authority_policy, 1, sqlite3_column_int(authority, 5));
    sqlite3_bind_int(d->authority_extensions, 1, sqlite3_column_int(authority, 0));

    if (sqlite3_step(d->authority_policy) == SQLITE_ROW)
      result.policy = d->toPolicy(d->authority_policy);
    else
      err = Database::RequestError;

    sqlite3_reset(d->authority_policy);

    // Fetch all extensions for authority.
    while ((extResult = sqlite3_step(d->authority_extensions)) == SQLITE_ROW)
      result.extensions << d->toExtension(d->authority_extensions);
    sqlite3_reset(d->authority_extensions);
    if (extResult != SQLITE_DONE)
      err = Database::RequestError;
  }
  else if (request == SQLITE_DONE)
    err = Database::NotFound;
  else {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    err = Database::RequestError;
  }

  sqlite3_finalize(authority);

  if (error)
    *error = err;

  return result;
}

Database::Error SQLiteDatabase::setAuthority(const Authority& authority)
{
  if (!d->handle)
    return Database::NotOpen;

  bool keyDirty = authority.keyDirty;
  sqlite3_exec(d->handle, "BEGIN IMMEDIATE TRANSACTION;", NULL, NULL, NULL);

  // Find policy with lowercase name
  int policy_id;
  sqlite3_stmt* policy;
  QByteArray policySql = QString("SELECT id FROM Policies WHERE LOWER(name) = '%1';")
                            .arg(authority.policy.name.toLower()).toUtf8();
  if (sqlite3_prepare_v2(d->handle, policySql.constData(), policySql.length(), &policy, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }
  if (sqlite3_step(policy) == SQLITE_ROW)
    policy_id = sqlite3_column_int(policy, 0);
  else {
    sqlite3_finalize(policy);
    QString policy_values = QString("Country = %1 AND STate = %2 AND Location = %3 AND "
                                    "Organization = %4 AND OrganizationalUnit = %5 AND "
                                    "CommonName = %6 AND Email = %7").arg(authority.policy.country)
                                .arg(authority.policy.state).arg(authority.policy.location)
                                .arg(authority.policy.organization).arg(authority.policy.organizationalUnit)
                                .arg(authority.policy.commonName).arg(authority.policy.email);
    policySql = QString("SELECT id FROM Policies "
                        "WHERE (%1);").arg(policy_values).toUtf8();
    if (sqlite3_prepare_v2(d->handle, policySql.constData(), policySql.length(), &policy, 0)) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::RequestError;
    }
    if (sqlite3_step(policy) == SQLITE_ROW)
      policy_id = sqlite3_column_int(policy, 0);
    else {
      policySql = QString("INSERT INTO Policies (name, Country, STate, Location,"
                                                "Organization, OrganizationalUnit, CommonName, Email)"
                            "VALUES ('%1', %2);").arg(authority.policy.name)
                                                 .arg(policy_values.replace(QRegExp("[^ ]+ = "), "")
                                                   .replace(" AND ", ", ")).toUtf8();
      sqlite3_exec(d->handle, policySql.constData(), NULL, NULL, NULL);
      policy_id = sqlite3_last_insert_rowid(d->handle);
      if (!policy_id) {
        d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
        sqlite3_finalize(policy);
        sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
        return Database::ConstraintError;
      }
    }
  }
  sqlite3_finalize(policy);

  int authority_id;
  sqlite3_stmt* db_authority;
  QByteArray authSql = QString("SELECT id FROM Authorities WHERE hash = %1;").arg(authority.hash).toUtf8();
  if (sqlite3_prepare_v2(d->handle, authSql.constData(), authSql.length(), &db_authority, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }
  if (sqlite3_step(db_authority) == SQLITE_ROW) {
    authority_id = sqlite3_column_int(db_authority, 0);
    authSql = QString("UPDATE Authorities SET name='%1', certificatedays=%2, policy=%3,"
                        "crlnumber=%4, crldays=%5, digest=%6 "
                      "WHERE id = %7;")
                  .arg(authority.name).arg(authority.certificateDays).arg(policy_id)
                  .arg(authority.crlnumber).arg(authority.crldays).arg(authority.digest).arg(authority_id).toUtf8();
    if (sqlite3_exec(d->handle, authSql.constData(), NULL, NULL, NULL) != SQLITE_OK) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_finalize(db_authority);
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::ConstraintError;
    }
  }
  else {
    sqlite3_finalize(db_authority);
    authSql = QString("INSERT INTO Authorities (hash, name, certificatedays, policy,"
                        "crlnumber, crldays, digest, certificate, key) "
                        "VALUES (%1, '%2', %3, %4, %5, %6, %7, @CERT, @KEY);")
                .arg(authority.hash).arg(authority.name).arg(authority.certificateDays)
                .arg(policy_id).arg(authority.crlnumber).arg(authority.crldays).arg(authority.digest).toUtf8();
    if (sqlite3_prepare_v2(d->handle, authSql.constData(), authSql.length(), &db_authority, 0)) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::RequestError;
    }
    sqlite3_bind_blob(db_authority, 1, authority.certificate, authority.certificate.length(), SQLITE_STATIC);
    sqlite3_bind_blob(db_authority, 2, authority.key, authority.key.length(), SQLITE_STATIC);

    if (sqlite3_step(db_authority) != SQLITE_DONE) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_finalize(db_authority);
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::ConstraintError;
    }
    keyDirty = false;
    authority_id = sqlite3_last_insert_rowid(d->handle);
  }
  sqlite3_finalize(db_authority);

  QByteArray ext_sql = QString("DELETE FROM CAExtensions WHERE authority = %1;").arg(authority_id).toUtf8();
  if (sqlite3_exec(d->handle, ext_sql.constData(), NULL, NULL, NULL) != SQLITE_OK) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }

  // Evaluate extension IDs.
  int ext_id;
  sqlite3_stmt *extension, *insert_ext;
  ext_sql = QString("SELECT id FROM Extensions "
                      "WHERE (type = ? AND critical = ? AND oid = ? AND value = ?);").toUtf8();
  if (sqlite3_prepare_v2(d->handle, ext_sql.constData(), ext_sql.length(), &extension, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }

  // Prepare extension insert
  ext_sql = QString("INSERT INTO CAExtensions (authority, extension) VALUES (%1, ?);").arg(authority_id).toUtf8();
  if (sqlite3_prepare_v2(d->handle, ext_sql.constData(), ext_sql.length(), &insert_ext, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }

  for (QList< Extension >::const_iterator it = authority.extensions.constBegin();
       it != authority.extensions.constEnd(); ++it) {
    ext_id = (*it).id;
    if (!ext_id) {
      sqlite3_bind_int(extension, 1,(int) (*it).type);
      sqlite3_bind_int(extension, 2, (*it).critical);
      sqlite3_bind_text(extension, 3, (*it).oid.toUtf8().constData(), -1, SQLITE_STATIC);
      sqlite3_bind_text(extension, 4, (*it).value.constData(), (*it).value.length(), SQLITE_STATIC);
      if (sqlite3_step(extension) == SQLITE_ROW)
        ext_id = sqlite3_column_int(extension, 0);
      else {
        ext_sql = QString("INSERT INTO Extensions (type, critical, oid, value)"
                          "VALUES (%1, %2, '%3', '%4');").arg((*it).type).arg((*it).critical)
                          .arg((*it).oid).arg(QString((*it).value)).toUtf8();
        sqlite3_exec(d->handle, ext_sql.constData(), NULL, NULL, NULL);
        ext_id = sqlite3_last_insert_rowid(d->handle);
        if (!ext_id) {
          d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
          sqlite3_finalize(extension);
          sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
          return Database::ConstraintError;
        }
      }
      sqlite3_reset(extension);
    }

    if (ext_id) {
      sqlite3_bind_int(insert_ext, 1, ext_id);
      if (sqlite3_step(insert_ext) != SQLITE_DONE) {
        d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
        sqlite3_finalize(extension);
        sqlite3_finalize(insert_ext);
        sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
        return Database::ConstraintError;
      }
      sqlite3_reset(insert_ext);
    }
  }
  sqlite3_finalize(extension);
  sqlite3_finalize(insert_ext);

  if (sqlite3_exec(d->handle, "COMMIT TRANSACTION;", NULL, NULL, NULL) != SQLITE_OK) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::TransactionError;
  }

  // Update authority key in database
  if (keyDirty) {
    sqlite3_stmt *update_key;
    QByteArray updateKey = QString("UPDATE Authorities SET key=@KEY WHERE id = %1;").arg(authority_id).toUtf8();
    if (!sqlite3_prepare_v2(d->handle, updateKey.constData(), updateKey.length(), &update_key, 0)) {
      sqlite3_bind_blob(update_key, 1, authority.key.constData(), authority.key.length(), SQLITE_STATIC);
      if (sqlite3_step(update_key) != SQLITE_DONE) {
        d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
        sqlite3_finalize(update_key);
        return Database::ConstraintError;
      }
      sqlite3_finalize(update_key);
      keyDirty = false;
    }
  }

  return Database::NoError;
}

Database::Error SQLiteDatabase::setAuthorityCertificate(const quint32 authorityhash,
                                                        const QByteArray& certificate, const QByteArray& key)
{
  if (!d->handle)
    return Database::NotOpen;
  if (certificate.isEmpty())
    return Database::ConstraintError;

  sqlite3_exec(d->handle, "BEGIN IMMEDIATE TRANSACTION;", NULL, NULL, NULL);

  sqlite3_stmt* update_cert;
  QByteArray authCertSql = QString("UPDATE Authorities SET certificate = @CERT WHERE hash = %1;").arg(authorityhash).toUtf8();
  if (sqlite3_prepare_v2(d->handle, authCertSql.constData(), authCertSql.length(), &update_cert, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }

  sqlite3_bind_blob(update_cert, 1, certificate.constData(), certificate.length(), SQLITE_STATIC);
  if (sqlite3_step(update_cert) != SQLITE_DONE) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    sqlite3_finalize(update_cert);
    return Database::RequestError;
  }
  sqlite3_finalize(update_cert);

  switch (sqlite3_changes(d->handle)) {
    case 0:
      return Database::NotFound;
      break;
    case 1: break;
    default:
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::ConstraintError;
      break;
  }

  if (!key.isEmpty()) {
    sqlite3_stmt* update_key;
    QByteArray authKeySql = QString("UPDATE Authorities SET key = @KEY WHERE hash = %1;").arg(authorityhash).toUtf8();
    if (sqlite3_prepare_v2(d->handle, authKeySql.constData(), authKeySql.length(), &update_key, 0)) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::RequestError;
    }

    sqlite3_bind_blob(update_key, 1, key.constData(), key.length(), SQLITE_STATIC);
    if (sqlite3_step(update_key) != SQLITE_DONE) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_finalize(update_key);
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::RequestError;
    }
    sqlite3_finalize(update_key);

    switch (sqlite3_changes(d->handle)) {
      case 0:
        return Database::NotFound;
        break;
      case 1: break;
      default:
        sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
        return Database::ConstraintError;
        break;
    }
  }

  if (sqlite3_exec(d->handle, "COMMIT TRANSACTION;", NULL, NULL, NULL) != SQLITE_OK) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::TransactionError;
  }

  return Database::NoError;
}

const QHash< QString, Policy > SQLiteDatabase::getPolicies(Database::Error* error) const
{
  QHash< QString, Policy > result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }
  if (error)
    *error = Database::NoError;

  static const QByteArray sql = QString("SELECT * FROM Policies;").toUtf8();
  if (!d->policies) {
    if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(), &d->policies, NULL) != SQLITE_OK) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      if (error)
        *error = Database::RequestError;
      return result;
    }
  }

  int request;
  while ((request = sqlite3_step(d->policies)) == SQLITE_ROW) {
    Policy policy = d->toPolicy(d->policies);
    result.insert(policy.name.toLower(), policy);
  }
  if (request != SQLITE_DONE && error)
    *error = Database::RequestError;

  return result;
}

const QList< Extension > SQLiteDatabase::getExtensions(Database::Error* error) const
{
  QList< Extension > result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }
  if (error)
    *error = Database::NoError;

  static const QByteArray sql = QString("SELECT * FROM Extensions;").toUtf8();
  if (!d->extensions) {
    if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(), &d->extensions, NULL) != SQLITE_OK) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      if (error)
        *error = Database::RequestError;
      return result;
    }
  }

  int request;
  while ((request = sqlite3_step(d->extensions)) == SQLITE_ROW)
    result << d->toExtension(d->extensions);
  if (request != SQLITE_DONE && error)
    *error = Database::RequestError;

  return result;
}

const QHash< quint64, Certificate > SQLiteDatabase::getCertificates(const quint32 authorityhash,
                                                                    Database::Error* error) const
{
  QHash< quint64, Certificate > result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }
  if (error)
    *error = Database::NoError;

  QByteArray sql;
  if (authorityhash)
    sql = QString("SELECT * FROM Certificates WHERE authority=%1;").arg(authorityhash).toUtf8();
  else
    sql = QString("SELECT * FROM Certificates;").toUtf8();

  sqlite3_stmt *certifcates;
  if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(), &certifcates, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    if (error)
      *error = Database::RequestError;
    return result;
  }

  int request;
  while ((request = sqlite3_step(certifcates)) == SQLITE_ROW) {
    Certificate certifcate = d->toCertificate(certifcates);
    result.insert(certifcate.serial, certifcate);
  }
  if (request != SQLITE_DONE && error)
    *error = Database::RequestError;

  sqlite3_finalize(certifcates);

  return result;
}

Certificate SQLiteDatabase::getCertificate(const QByteArray& fingerprint, Database::Error* error) const
{
  Certificate result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }

  Database::Error err = Database::NoError;

  QByteArray sql = QString("SELECT * FROM Certificates WHERE fingerprint='%1';").arg(QString(fingerprint)).toUtf8();
  sqlite3_stmt *certifcate;
  if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(), &certifcate, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    if (error)
      *error = Database::RequestError;
    return result;
  }

  int request = sqlite3_step(certifcate);
  if (request == SQLITE_ROW)
    result = d->toCertificate(certifcate);
  else if (request == SQLITE_DONE)
    err = Database::NotFound;
  else {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    err = Database::RequestError;
  }

  sqlite3_finalize(certifcate);

  if (error)
    *error = err;

  return result;
}

Certificate SQLiteDatabase::getCertificate(const quint32 authorityhash, const quint64 serial,
                                           Database::Error* error) const
{
  Certificate result;
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return result;
  }

  if (!d->certificate) {
    if (error)
      *error = Database::PreparationError;
    return result;
  }

  Database::Error err = Database::NoError;
  sqlite3_bind_int64(d->certificate, 1, authorityhash);
  sqlite3_bind_int64(d->certificate, 2,(qint64) serial);

  int request = sqlite3_step(d->certificate);
  if (request == SQLITE_ROW)
    result = d->toCertificate(d->certificate);
  else if (request == SQLITE_DONE)
    err = Database::NotFound;
  else {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    err = Database::RequestError;
  }

  sqlite3_reset(d->certificate);

  if (error)
    *error = err;

  return result;
}

Database::Error SQLiteDatabase::setCertificate(const Certificate& certificate)
{
  if (!d->handle)
    return Database::NotOpen;

  if (certificate.expires < certificate.issued)
    return Database::ConstraintError;

  sqlite3_exec(d->handle, "BEGIN IMMEDIATE TRANSACTION;", NULL, NULL, NULL);

  int certificate_id;
  sqlite3_stmt* db_certificate;
  QByteArray certSql = QString("SELECT id FROM Certificates WHERE authority = %1 AND serial = %2;")
                          .arg(certificate.authorityhash).arg((qint64) certificate.serial).toUtf8();
  if (sqlite3_prepare_v2(d->handle, certSql.constData(), certSql.length(), &db_certificate, 0)) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::RequestError;
  }
  if (sqlite3_step(db_certificate) == SQLITE_ROW) {
    certificate_id = sqlite3_column_int(db_certificate, 0);
    certSql = QString("UPDATE Certificates SET owner='%1', status=%2, revocation='%3', reason=%4 "
                      "WHERE (status = 0 OR (status = 2 AND reason = %5)) AND id = %6;")
                  .arg(certificate.owner).arg(certificate.state)
                  .arg(certificate.revoked.toUTC().toString(Qt::ISODate)).arg(certificate.reason)
                  .arg(kCA_ossl::CertificateHold).arg(certificate_id).toUtf8();
    if (sqlite3_exec(d->handle, certSql.constData(), NULL, NULL, NULL) != SQLITE_OK) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_finalize(db_certificate);
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::RequestError;
    }
  }
  else {
    sqlite3_finalize(db_certificate);
    certSql = QString("INSERT INTO Certificates (serial, fingerprint, authority, subject, issued, validfrom, expiration,"
                          "owner, status, revocation, reason, certificate) "
                      "VALUES (@SERIAL, @FP, %1, '%2', '%3', '%4', '%5', '%6', %7, '%8', %9, @CERT);")
                  .arg(certificate.authorityhash)
                  .arg(certificate.subject, certificate.issued.toUTC().toString(Qt::ISODate),
                       certificate.validFrom.toUTC().toString(Qt::ISODate),
                       certificate.expires.toUTC().toString(Qt::ISODate), certificate.owner)
                  .arg(certificate.state).arg(certificate.revoked.toUTC().toString(Qt::ISODate))
                  .arg(certificate.reason).toUtf8();
    if (sqlite3_prepare_v2(d->handle, certSql.constData(), certSql.length(), &db_certificate, 0)) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
      return Database::RequestError;
    }
    sqlite3_bind_int64(db_certificate, 1,(qint64) certificate.serial);
    sqlite3_bind_text(db_certificate, 2, certificate.fingerprint, certificate.fingerprint.length(), SQLITE_STATIC);
    sqlite3_bind_blob(db_certificate, 3, certificate.certificate, certificate.certificate.length(), SQLITE_STATIC);
  }

  if (sqlite3_step(db_certificate) != SQLITE_DONE) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_finalize(db_certificate);
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::ConstraintError;
  }
  sqlite3_finalize(db_certificate);

  if (sqlite3_exec(d->handle, "COMMIT TRANSACTION;", NULL, NULL, NULL) != SQLITE_OK) {
    d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
    sqlite3_exec(d->handle, "ROLLBACK TRANSACTION;", NULL, NULL, NULL);
    return Database::TransactionError;
  }

  return Database::NoError;
}

bool SQLiteDatabase::isSerialUsed(const quint32 authorityhash, const quint64 serial, Database::Error* error) const
{
  if (!d->handle) {
    if (error)
      *error = Database::NotOpen;
    return true;
  }
  if (error)
    *error = Database::NoError;

  QByteArray sql = QString("SELECT 1 FROM Certificates WHERE authority=$AUTHORITY AND serial=$SERIAL;").toUtf8();
  if (!d->authority_serial) {
    if (sqlite3_prepare_v2(d->handle, sql.constData(), sql.length(),
                           &d->authority_serial, NULL) != SQLITE_OK) {
      d->error = QString("%1: %2").arg(sqlite3_errcode(d->handle)).arg(sqlite3_errmsg(d->handle));
      if (error)
        *error = Database::RequestError;

      return true;
    }
  }

  sqlite3_bind_int64(d->authority_serial, 1, authorityhash);
  sqlite3_bind_int64(d->authority_serial, 2,(sqlite3_int64) serial);
  bool result = (sqlite3_step(d->authority_serial) != SQLITE_DONE);

  sqlite3_reset(d->authority_serial);

  return result;
}
