/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010, 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "authority.h"

#include "database.h"

#include <libkca_ossl/certificate.h>
#include <libkca_ossl/commons.h>
#include <libkca_ossl/request.h>
#include <libkca_ossl/revocationlist.h>
#include <libkca_ossl/x509extension.h>
#include <libkca_ossl/x509name.h>

#include <QtCore/QByteArray>
#include <QtCore/QDateTime>
#include <QtCore/QFile>
#include <QtCore/QHash>
#include <QtCore/QList>
#include <QtCore/QLocale>
#include <QtCore/QSettings>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QTime>
#include <QtCore/QUrl>

#include <sys/stat.h>

namespace kCA
{

class AuthorityPrivate
{
public:
  AuthorityPrivate(Database::Database* database) :
      revoked(false),
      dirty(false),
      keyLoaded(false),
      keepKey(false),
      lockableKey(true),
      db(database),
      crlDistributionPoint(QUrl()),
      certExtensions(QList<const kCA_ossl::X509Extension*>()),
      crlExtensions(QList<const kCA_ossl::X509Extension*>())
  {
    internal.extensions = QList<Database::Extension>();
  }
  AuthorityPrivate(Database::Database* database, const quint32 hash) :
      revoked(false),
      dirty(false),
      keyLoaded(false),
      keepKey(false),
      lockableKey(true),
      db(database)
  {
    internal = db->getAuthority(hash);
    certificate = kCA_ossl::Certificate::fromRawData(internal.certificate);
    subject = certificate.subject();

    certExtensions.clear();
    crlExtensions.clear();
    foreach (Database::Extension ext, internal.extensions) {
      switch (ext.type) {
        case Database::Extension::Certificate:
          certExtensions.append(new kCA_ossl::X509Extension(ext.oid, ext.value, ext.critical));
          if (ext.oid == "2.5.29.31") {
            QByteArray value = ext.value;
            if (value.startsWith("URI:"))
              value.remove(0, 4);
            crlDistributionPoint = QUrl(value);
          }
          break;
        case Database::Extension::CRL:
          crlExtensions.append(new kCA_ossl::X509Extension(ext.oid, ext.value, ext.critical));
          break;
      }
    }

    certificates = db->getCertificates(hash);

    Database::Database::Error err;
    Database::Certificate cert = database->getCertificate(certificate.fingerprint(), &err);
    if (err == Database::Database::NoError)
      revoked = (cert.state == Database::Certificate::Revoked);
  }
  ~AuthorityPrivate()
  {
    if (dirty)
      db->setAuthority(internal);
  }

  bool policyMatched(const kCA_ossl::X509Name& other)
  {
    static const QStringList keys(QStringList() << "C" << "ST" << "L" << "O" << "OU" << "CN" << "ITU-T");

    Database::Policy::FieldSetting policy;
    foreach (QString key, keys) {
      if (key == "C")
        policy = internal.policy.country;
      else if (key == "ST")
        policy = internal.policy.state;
      else if (key == "L")
        policy = internal.policy.location;
      else if (key == "O")
        policy = internal.policy.organization;
      else if (key == "OU")
        policy = internal.policy.organizationalUnit;
      else if (key == "CN")
        policy = internal.policy.commonName;
      else if (key == "ITU-T") // || key.toLower() == "emailaddress")
        policy = internal.policy.email;
      else
        continue;

      switch (policy) {
        case Database::Policy::Optional:
          continue;
          break;
        case Database::Policy::Supplied:
          if (!other.keys().contains(key))
            return false;
          break;
        case Database::Policy::Match:
          int length = subject.values(key).length();
          if (other.keys().contains(key) && other.values(key).length() == length) {
            QStringList localValues = subject.values(key);
            QStringList otherValues = other.values(key);
            for (int i = 0; i < length; ++i)
              if (otherValues.at(i) != localValues.at(i))
                return false;
          }
          else
            return false;

          break;
      }
    }

    return true;
  }

  bool revoked;
  bool dirty;
  bool keyLoaded, keepKey, lockableKey;

  Database::Database* db;
  Database::Authority internal;

  kCA_ossl::Certificate certificate;
  kCA_ossl::X509Name subject;

  QUrl crlDistributionPoint;

  QList<const kCA_ossl::X509Extension*> certExtensions;
  QList<const kCA_ossl::X509Extension*> crlExtensions;

  QHash< quint64, Database::Certificate > certificates;

  QString backendError;
};

};

using namespace kCA;

Authority::Authority(Database::Database* database, const quint32 hash, QObject* parent) :
    QObject(parent),
    d(new AuthorityPrivate(database, hash))
{
  // Try to auto-unlock if unencrypted key is stored in database.
  if (keyPath().isEmpty())
    d->lockableKey = ~unlockKey(QString(), true);
}

Authority::Authority(Database::Database* database, const QString& name, const Database::Policy& policy,
                     const kCA_ossl::Certificate& certificate,
                     const QByteArray& key, QObject* parent) :
    QObject(parent),
    d(new AuthorityPrivate(database))
{
  d->internal.name = name;
  d->internal.certificateDays = 365;
  d->internal.crlnumber = 0;
  d->internal.crldays = 7;
  d->internal.digest = certificate.digest();
  d->internal.policy = policy;
  d->internal.hash = certificate.subjectHash();
  d->internal.certificate = certificate.toPem();
  d->internal.key = key;

  d->certificate = certificate;
  d->subject = certificate.subject();

  d->db->setAuthority(d->internal);

  Database::Database::Error err;
  Database::Certificate cert = database->getCertificate(certificate.fingerprint(), &err);
  if (err == Database::Database::NoError)
    d->revoked = (cert.state == Database::Certificate::Revoked);
}

Authority::~Authority()
{
  delete d;
}

bool Authority::enabled() const
{
  if (d->revoked)
    return false;

  QDateTime now = QDateTime::currentDateTime();
  if (now < d->certificate.notBefore() || now > d->certificate.notAfter())
    return false;

  return true;
}

quint32 Authority::hash() const
{
  return d->internal.hash;
}

quint32 Authority::issuerHash() const
{
  return d->certificate.issuerHash();
}

QString Authority::name() const
{
  return d->internal.name;
}

void Authority::setName(const QString& value)
{
  d->dirty = d->dirty | (d->internal.name != value);
  d->internal.name = value;
}

QString Authority::subject() const
{
  return d->certificate.subject().toString();
}

unsigned int Authority::certificateDays() const
{
  return d->internal.certificateDays;
}

void Authority::setCertificateDays(unsigned int value)
{
  d->dirty = d->dirty | (d->internal.certificateDays != value);
  d->internal.certificateDays = value;
}

unsigned int Authority::crlDays() const
{
  return d->internal.crldays;
}

void Authority::setCrlDays(unsigned int value)
{
  d->dirty = d->dirty | (d->internal.crldays != value);
  d->internal.crldays = value;
}

kCA_ossl::Digest Authority::digest() const
{
  return d->internal.digest;
}

void Authority::setDigest(kCA_ossl::Digest digest)
{
  d->dirty = d->dirty | (d->internal.digest != digest);
  d->internal.digest = digest;
}

QUrl Authority::crlDistributionPoint() const
{
  return d->crlDistributionPoint;
}

void Authority::setCrlDistributionPoint(const QUrl& value)
{
  if (!d->crlDistributionPoint.isEmpty())
    return;

  QByteArray encoded_url = value.toEncoded(QUrl::RemoveUserInfo);
  if (encoded_url.isEmpty())
    return;

  d->dirty = true;
  d->crlDistributionPoint = value;
  d->certExtensions.append(new kCA_ossl::X509Extension("2.5.29.31", encoded_url));

  Database::Extension crlDpExt;
  crlDpExt.id = 0;
  crlDpExt.type = Database::Extension::Certificate;
  crlDpExt.oid = "2.5.29.31";
  crlDpExt.critical = false;
  crlDpExt.value = encoded_url;
  d->internal.extensions.append(crlDpExt);
}

Database::Policy Authority::getPolicy() const
{
  return d->internal.policy;
}

void Authority::setPolicy(const Database::Policy& policy)
{
  d->dirty = true;
  d->internal.policy = policy;
}

QList< const kCA_ossl::X509Extension* > Authority::certificateExtensions() const
{
  return d->certExtensions;
}

void Authority::setCertificateExtensions(const QList< const kCA_ossl::X509Extension* >& extensions)
{
  d->dirty = true;
  d->certExtensions = extensions;

  // Remove all certificate-related extensions
  for (int i = 0; i < d->internal.extensions.count(); ++i)
    if (d->internal.extensions.at(i).type == Database::Extension::Certificate)
      d->internal.extensions.removeAt(i);

  Database::Extension dbext;
  dbext.id = 0;
  dbext.type = Database::Extension::Certificate;
  foreach (const kCA_ossl::X509Extension* ext, extensions) {
    if (!ext->isValid())
      continue;

    if (ext->oid() == "2.5.29.31") {
      if (!d->crlDistributionPoint.isEmpty())
        continue;
      else
        d->crlDistributionPoint = QUrl(ext->value());
    }

    dbext.oid = ext->oid();
    dbext.critical = ext->isCritical();
    dbext.value = ext->value();
    d->internal.extensions.append(dbext);
  }
}

QList< const kCA_ossl::X509Extension* > Authority::crlExtensions() const
{
  return d->crlExtensions;
}

void Authority::setCrlExtensions(const QList< const kCA_ossl::X509Extension* >& extensions)
{
  d->dirty = true;
  d->crlExtensions = extensions;

  // Remove all CRL-related extensions
  for (int i = 0; i < d->internal.extensions.count(); ++i)
    if (d->internal.extensions.at(i).type == Database::Extension::CRL)
      d->internal.extensions.removeAt(i);

  Database::Extension dbext;
  dbext.id = 0;
  dbext.type = Database::Extension::CRL;
  foreach (const kCA_ossl::X509Extension* ext, extensions) {
    if (!ext->isValid())
      continue;

    dbext.oid = ext->oid();
    dbext.critical = ext->isCritical();
    dbext.value = ext->value();
    d->internal.extensions.append(dbext);
  }
}

const QHash< quint64, Database::Certificate > Authority::certificates() const
{
  return d->certificates;
}

bool Authority::matchPolicy(const kCA_ossl::X509Name& name) const
{
  return d->policyMatched(name);
}

bool Authority::lockableKey() const
{
  return d->lockableKey;
}

bool Authority::unlockKey(const QString& password, const bool keep,
                          const QString& location, const bool store)
{
  if (d->keyLoaded)
    return true;

  bool loaded = false;
  if (!location.isEmpty()) {
    loaded = d->certificate.loadPrivateKey(location, password);
    if (loaded && store) {
      d->internal.key = location.toUtf8();
    }
  }
  else {
    if (keyPath().isEmpty())
      loaded = d->certificate.addPrivateKey(d->internal.key, password);
    else
      loaded = d->certificate.loadPrivateKey(d->internal.key, password);
  }

  if (loaded) {
    d->keepKey = keep;
    emit keyUnlocked(d->internal.hash);
  }

  return (d->keyLoaded = loaded);
}

void Authority::lockKey()
{
  if (!d->lockableKey)
    return;

  d->certificate.unloadPrivateKey();
  d->keyLoaded = false;
  emit keyLocked(d->internal.hash);
}

bool Authority::keyUnlocked() const
{
  return d->keyLoaded;
}

QString Authority::keyPath() const
{
  if (d->internal.key.startsWith("-----BEGIN RSA PRIVATE KEY-----\n"))
    return QString();
  else
    return QString(d->internal.key).trimmed();
}

bool Authority::replaceCertificate(const kCA_ossl::Certificate& certificate,
                                   const QByteArray& key, const QString& passphrase)
{
  if (!d->keyLoaded)
    return false;
  if (!certificate.isValid() || !certificate.isCA())
    return false;
  if (certificate.subjectHash() != d->certificate.subjectHash())
    return false;

  bool keyLoaded = false;
  kCA_ossl::Certificate cert(certificate);
  if (key.startsWith("-----BEGIN RSA PRIVATE KEY-----\n"))
    keyLoaded = cert.addPrivateKey(key, passphrase);
  else
    keyLoaded = cert.loadPrivateKey(key, passphrase);

  if (keyLoaded) {
    if (d->db->setAuthorityCertificate(d->internal.hash, cert.toPem(), key) != kCA::Database::Database::NoError)
      return false;

    d->certificate = cert;
    d->internal.key = key;
    return true;
  }

  return false;
}

QString Authority::databaseError() const
{
  return d->backendError;
}

void Authority::setCrlNumber(ulong value)
{
  d->internal.crlnumber = value;
  d->dirty = true;
}

void Authority::signRequest(const QString& owner, const kCA_ossl::Request& request,
                            const QDateTime& starttime, const QDateTime& endtime,
                            const QList< const kCA_ossl::X509Extension*>& extensions,
                            const bool supersede, const unsigned int timeout)
{
  kCA_ossl::Certificate empty;
  if (!d->keyLoaded) {
    emit requestSigned(empty, KeyNotLoaded);
    return;
  }

  if (endtime < starttime) {
    emit requestSigned(empty, ConstraintError);
    if (!d->keepKey)
      lockKey();
    return;
  }

  // Check if policy is matched
  if (!d->policyMatched(request.subject())) {
    emit requestSigned(empty, ConstraintError);
    if (!d->keepKey)
      lockKey();
    return;
  }

  Database::Certificate dbcert;
  dbcert.authorityhash = d->internal.hash;
  dbcert.state = Database::Certificate::Valid;

  QDateTime notBefore = QDateTime::currentDateTime().toUTC();
  dbcert.issued = notBefore;
  dbcert.expires = QDateTime::currentDateTime().addDays(d->internal.certificateDays).toUTC();
  if (starttime.isValid())
    notBefore = starttime;
  if (endtime.isValid())
    dbcert.expires = endtime;

  Database::Database::Error error;
  quint64 serial;
  QTime timer;
  timer.start();
  while (d->db->isSerialUsed(d->internal.hash, (serial = kCA_ossl::generateRandom()), &error)) {
    if (serial ==(quint64) ~0) {
      emit requestSigned(empty, RandomizerError);
      break;
    }
    if (error != Database::Database::NoError) {
      d->backendError = d->db->lastError();
      emit requestSigned(empty, DatabaseError);
      serial =(quint64) ~0;
      break;
    }
    if ((unsigned int) timer.elapsed() > timeout*1000) {
      emit requestSigned(empty, TimeoutError);
      serial =(quint64) ~0;
      break;
    }
  }

  QList< const kCA_ossl::X509Extension* > allExtensions(extensions);
  for (QList< const kCA_ossl::X509Extension* >::const_iterator it = d->certExtensions.constBegin();
       it != d->certExtensions.constEnd(); ++it) {
    bool found = false;
    for (QList< const kCA_ossl::X509Extension* >::const_iterator extIt = extensions.constBegin();
         extIt != extensions.constEnd(); ++extIt) {
      if ((*it)->oid() == (*extIt)->oid()) {
        found = true;
        break;
      }
    }
    if (found)
      continue;
    else
      allExtensions.append((*it));
  }

  if (serial !=(quint64) ~0) {
    kCA_ossl::Certificate cert = d->certificate.sign(request, serial, d->internal.digest,
                                                     notBefore, dbcert.expires, allExtensions);
    if (cert.issuerHash() == d->internal.hash) {
      dbcert.serial = serial;
      dbcert.owner = owner;
      dbcert.validFrom = notBefore;
      dbcert.fingerprint = cert.fingerprint();
      dbcert.subject = cert.subject().toString();
      dbcert.certificate = cert.toPem();

      error = d->db->setCertificate(dbcert);
      if (error == Database::Database::NoError) {
        d->certificates.insert(serial, dbcert);
        emit requestSigned(cert, NoError);
        if (supersede) {
          bool keepKey = d->keepKey;
          d->keepKey = true;
          foreach (Database::Certificate entry, d->certificates) {
            if (entry.serial != dbcert.serial
                && entry.state == Database::Certificate::Valid
                && entry.expires > dbcert.validFrom) {
              kCA_ossl::Certificate ossl_cert(entry.certificate);
              if (cert.subject().hash() == ossl_cert.subject().hash()) {
                revokeCertificate(entry.serial, kCA_ossl::Superseded, dbcert.validFrom);
              }
            }
          }
          d->keepKey = keepKey;
        }
      }
      else {
        d->backendError = d->db->lastError();
        emit requestSigned(empty, DatabaseError);
      }
    }
    else {
      emit requestSigned(empty, SigningError);
    }
  }

  if (!d->keepKey)
    lockKey();
}

void Authority::revokeCertificate(const quint64 serial, const kCA_ossl::RevocationReason reason,
                                  const QDateTime& timestamp)
{
  if (!d->keyLoaded) {
    emit certificateRevoked(d->internal.hash, serial,(int) reason, KeyNotLoaded);
    return;
  }

  Database::Database::Error error;
  Database::Certificate cert = d->db->getCertificate(d->internal.hash, serial, &error);
  if (error == Database::Database::NotFound) {
    emit certificateRevoked(d->internal.hash, serial,(int) reason, NotFound);
    if (!d->keepKey)
      lockKey();
    return;
  }
  if (cert.state == Database::Certificate::Expired || cert.expires <= QDateTime::currentDateTime().toUTC()) {
    emit certificateRevoked(d->internal.hash, serial,(int) reason, ConstraintError);
    if (!d->keepKey)
      lockKey();
    return;
  }

  if (reason == kCA_ossl::RemoveFromCRL) {
    if (cert.reason == kCA_ossl::CertificateHold) {
      cert.revoked = QDateTime();
      cert.state = Database::Certificate::Valid;
    }
    else {
      emit certificateRevoked(d->internal.hash, serial,(int) reason, ConstraintError);
      if (!d->keepKey)
        lockKey();
      return;
    }
  }
  else {
    cert.state = Database::Certificate::Revoked;
    cert.reason = reason;
    if (timestamp.isValid())
      cert.revoked = timestamp.toUTC();
    else
      cert.revoked = QDateTime::currentDateTime().toUTC();
  }

  if (d->db->setCertificate(cert) == Database::Database::NoError)
    emit certificateRevoked(d->internal.hash, serial,(int) reason, NoError);
  else {
    d->backendError = d->db->lastError();
    emit certificateRevoked(d->internal.hash, serial,(int) reason, DatabaseError);
  }

  if (!d->keepKey)
    lockKey();
}

void Authority::generateCrl(const unsigned int days,
                            const QList< const kCA_ossl::X509Extension* >& extensions)
{
  kCA_ossl::Revocationlist empty;

  if (!d->keyLoaded) {
    emit crlGenerated(empty, KeyNotLoaded);
    return;
  }

  kCA_ossl::Revocationlist crl(d->certificate, ++d->internal.crlnumber);
  // One hour latency to allow for manual CRL updates which do not happen on time.
  crl.setNextUpdate(QDateTime::currentDateTime().addDays(days > 0 ? days : d->internal.crldays).addSecs(3600));
  crl.setDistributionPoint(d->crlDistributionPoint);

  QList<const kCA_ossl::X509Extension*> exts;
  if (extensions.isEmpty())
    exts = d->crlExtensions;
  else
    exts = extensions;
  foreach (const kCA_ossl::X509Extension* ext, exts)
    crl.addExtension(*ext);

  Database::Database::Error err;
  QList< Database::Certificate > certs = d->db->getCertificates(d->internal.hash, &err).values();
  if (err != Database::Database::NoError) {
    emit crlGenerated(empty, DatabaseError);
    if (!d->keepKey)
      lockKey();
    return;
  }

  foreach (Database::Certificate cert, certs) {
    if (cert.state != Database::Certificate::Revoked)
      continue;

    kCA_ossl::Certificate crt(cert.certificate);
    crl.addEntry(crt, cert.reason, cert.revoked);
  }

  if (d->db->setAuthority(d->internal) != Database::Database::NoError) {
    d->backendError = d->db->lastError();
    emit crlGenerated(empty, DatabaseError);
    if (!d->keepKey)
      lockKey();
    return;
  }

  crl = d->certificate.sign(crl, d->internal.digest);
  if (!d->keepKey)
    lockKey();

  emit crlGenerated(crl, NoError);
}

QStringList Authority::findAuthorities(const QString& configfile)
{
  QStringList result;
  QSettings conf(configfile, QSettings::IniFormat);
  foreach (QString section, conf.childGroups()) {
    conf.beginGroup(section);
      if (conf.childKeys().contains("private_key"))
        result.append(section);
    conf.endGroup();
  }

  return result;
}

bool Authority::importAuthority(Database::Database* database, const QString& configfile,
                                const QString& name, const bool importKey)
{
  if (!database->isOpen())
    return false;

  QSettings conf(configfile, QSettings::IniFormat);
  if (conf.status() != QSettings::NoError)
    return false;

  QHash< QString, QString > globals, ca;
  foreach (QString key, conf.childKeys()) {
    if (!key.startsWith('#'))
      globals.insert(key, conf.value(key).toString());
  }

  QRegExp var("(\\$\\{(.+)\\})|(\\$([a-z0-9]+))");
  QString value, replacement, varname;

  conf.beginGroup(name);
  foreach (QString key, conf.childKeys()) {
    if (key.startsWith('#'))
      continue;

    value = conf.value(key).toString();

    int pos = 0;
    if ((pos = value.indexOf('#')) > -1)
      value.truncate(pos);

    while ((pos = value.indexOf(var)) > -1) {
      if (var.cap(2).isEmpty())
        varname = var.cap(4);
      else
        varname = var.cap(2);

      replacement = conf.value(varname).toString();
      if (replacement.isEmpty())
        replacement = globals.value(varname);

      int commentPos = 0;
      if ((commentPos = replacement.indexOf('#')) > -1)
        replacement.truncate(commentPos);

      value.replace(pos, varname.length()+1, replacement.trimmed());
    }

    ca.insert(key, value.trimmed());
  }
  conf.endGroup();

  if (!ca.count())
    return false;

  Database::Policy policy = database->getPolicies().value("default");
  if (ca.contains("policy")) {
    policy.name = ca.value("policy");
    conf.beginGroup(policy.name);
    policy.country = Database::Policy::convert(conf.value("countryName").toString());
    policy.state = Database::Policy::convert(conf.value("stateOrProvinceName").toString());
    policy.location = Database::Policy::convert(conf.value("localityName").toString());
    policy.organization = Database::Policy::convert(conf.value("organizationName").toString());
    policy.organizationalUnit = Database::Policy::convert(conf.value("organizationalUnitName").toString());
    policy.commonName = Database::Policy::convert(conf.value("commonName").toString());
    policy.email = Database::Policy::convert(conf.value("emailAddress").toString());
    conf.endGroup();
    policy.name.remove("policy_");
  }

  QByteArray key = ca.value("private_key").toUtf8();

  QFile certificate(ca.value("certificate"));
  if (!certificate.exists())
    return false;

  if (importKey) {
    QFile keyFile(ca.value("private_key"));
    if (!keyFile.exists())
      return false;

    keyFile.open(QFile::ReadOnly);
    key = keyFile.readAll();
    keyFile.close();
  }

  certificate.open(QFile::ReadOnly);
  Authority authority(database, name, policy, certificate.readAll(), key);
  certificate.close();

  authority.setCertificateDays(ca.value("default_days").toInt());
  authority.setCrlDays(ca.value("default_crl_days").toInt());

  // Load current CRL number
  QFile crlNumberFile(ca.value("crlnumber"));
  if (crlNumberFile.exists()) {
    crlNumberFile.open(QFile::ReadOnly);
    ulong crlnumber = crlNumberFile.readLine().trimmed().toULong(0, 16);
    crlNumberFile.close();
    if (crlnumber > 0)
      authority.setCrlNumber(crlnumber);
  }

  // Store digest setting
  QString source_md = ca.value("default_md").toLower();
  if (source_md != "default") {
    if (source_md == "ripemd160")
      authority.setDigest(kCA_ossl::RIPEMD160);
    else if (source_md == "sha256")
      authority.setDigest(kCA_ossl::SHA256);
    else if (source_md == "sha384")
      authority.setDigest(kCA_ossl::SHA384);
    else if (source_md == "sha512")
      authority.setDigest(kCA_ossl::SHA512);
    else
      // No support for MD5 and derivates so default to SHA1 if an unknown digest is specified
      authority.setDigest(kCA_ossl::SHA1);
  }

  // Load extensions used for this CA
  QRegExp critical("critical", Qt::CaseInsensitive);
  QList< const kCA_ossl::X509Extension* > extensions;
  conf.beginGroup(ca.value("x509_extensions"));
  foreach (QString extKey, conf.childKeys()) {
    if (extKey.startsWith('#'))
      continue;

    bool criticality = false;
    QStringList extValues = conf.value(extKey).toStringList();
    if (extValues.contains("critical", Qt::CaseInsensitive)) {
      int pos = 0;
      while ((pos = extValues.indexOf(critical)) > -1)
        extValues.removeAt(pos);

      criticality = true;
    }
    extensions.append(new kCA_ossl::X509Extension(extKey, extValues.join(", ").toUtf8(), criticality));
  }
  conf.endGroup();
  authority.setCertificateExtensions(extensions);

  // Load certificates
  QFile cadb(ca.value("database"));
  if (!cadb.exists())
    return false;

  struct stat buffer;
  Database::Certificate cert;
  cert.authorityhash = authority.hash();
  cert.owner = "";

  // Self-signed certificates must also be loaded as certificate
  if (authority.issuerHash() == authority.hash()) {
    if (stat(ca.value("certificate").toUtf8().constData(), &buffer))
      cert.issued = authority.d->certificate.notBefore();
    else {
      cert.issued = QDateTime::fromTime_t(buffer.st_ctim.tv_sec);
      cert.issued.setTimeSpec(Qt::LocalTime);
      cert.issued = cert.issued.toUTC();
    }

    cert.state = Database::Certificate::Valid;
    cert.revoked = QDateTime();
    cert.certificate = authority.d->internal.certificate;
    cert.validFrom = authority.d->certificate.notBefore();
    cert.expires = authority.d->certificate.notAfter();
    cert.fingerprint = authority.d->certificate.fingerprint();
    cert.serial = authority.d->certificate.serial();
    cert.subject = authority.subject();
    database->setCertificate(cert);
  }

  cadb.open(QFile::ReadOnly);
  while (!cadb.atEnd()) {
    QList< QByteArray > line = cadb.readLine().trimmed().split('\t');
    QString x509file = QString("%1/%2.pem").arg(ca.value("new_certs_dir")).arg(QString(line.at(3)));
    kCA_ossl::Certificate x509certificate(x509file);
    // Certificate file not found, proceed with next
    if (!x509certificate.serial())
      continue;

    if (stat(x509file.toUtf8().constData(), &buffer))
      cert.issued = x509certificate.notBefore();
    else {
      cert.issued = QDateTime::fromTime_t(buffer.st_ctim.tv_sec);
      cert.issued.setTimeSpec(Qt::LocalTime);
      cert.issued = cert.issued.toUTC();
    }

    cert.state = Database::Certificate::Valid;
    cert.revoked = QDateTime();
    cert.certificate = x509certificate.toPem();
    cert.validFrom = x509certificate.notBefore();
    cert.expires = x509certificate.notAfter();
    cert.fingerprint = x509certificate.fingerprint();
    cert.serial = x509certificate.serial();
    cert.subject = line.at(5);

    // Add revocation information if any
    if (line.at(0) == "R") {
      cert.state = Database::Certificate::Revoked;
      QStringList revocationData = QString(line.at(2)).split(',');
      cert.revoked = QLocale::c().toDateTime(QString("20%1").arg(revocationData.at(0)), "yyyyMMddHHmmss'Z'");
      cert.revoked.setTimeSpec(Qt::UTC);
      QString reason = revocationData.at(1).toLower();
      if (reason == "keycompromise")
        cert.reason = kCA_ossl::KeyCompromise;
      else if (reason == "cacompromise")
        cert.reason = kCA_ossl::CACompromise;
      else if (reason == "affilitionchanged")
        cert.reason = kCA_ossl::AffilitionChanged;
      else if (reason == "superseded")
        cert.reason = kCA_ossl::Superseded;
      else if (reason == "cessationofoperation")
        cert.reason = kCA_ossl::CessationOfOperation;
      else if (reason == "certificatehold")
        cert.reason = kCA_ossl::CertificateHold;
      else
        cert.reason = kCA_ossl::Unspecified;
    }
    database->setCertificate(cert);
  }
  cadb.close();

  return true;
}

#include "authority.moc"
