/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _LIBKCAOSSL_X509EXTENSION_H
#define _LIBKCAOSSL_X509EXTENSION_H

#include "kca_ossl_export.h"

#include <QtCore/QObject>

struct X509_extension_st;
typedef struct X509_extension_st X509_EXTENSION;

class QString;

/** Namespace for kca-local classes */
namespace kCA_ossl
{

class Certificate;
class CertificatePrivate;
class Request;
class RequestPrivate;
class Revocationlist;
class RevocationlistPrivate;

/**
 * Represents a X509 extension for certificates and requests.
 * Basically adds an abstraction layer over OpenSSL X509_EXTENSION.
 *
 * @short OpenSSL X509_EXTENSION abstraction class.
 * @author Felix Tiede <info@pc-tiede.de>
 */
class _LIBKCAOSSL_EXPORT X509Extension : public QObject
{
  friend class Certificate;
  friend class CertificatePrivate;
  friend class Request;
  friend class RequestPrivate;
  friend class Revocationlist;
  friend class RevocationlistPrivate;

  Q_OBJECT
  public:
    /** Creates an empty X509 extension. */
    X509Extension(QObject *parent = 0);
    /** Copy constructor. */
    X509Extension(const X509Extension &other, QObject *parent = 0);
    /**
     * Creates a X509 extension.
     *
     * @param oid Object identifier for the extension. Must be known to OpenSSL.
     * @param value Value to store for this extension.
     * @param critical set to true, if extension is critical. Default is false.
     * @param parent Owner object of this instance, default 0.
     */
    X509Extension(const QString &oid, const QByteArray &value,
                  bool critical = false, QObject *parent = 0);
    /**
     * Creates an extension for a X509 certificate or request.
     * It also adds new OIDs to OpenSSL's internal table, so parsing future extensions
     * forth and back to OpenSSL's internal format can rely on OpenSSL if an OID
     * is valid or not.
     *
     * @param oid Object identifier for the extension.
     * @param oidShortName Short name of object identifier.
     * @param oidLongName Long name of object identifier.
     * @param value Value to store for this extension.
     * @param critical set to true, if extension is critical. Default is false.
     * @param parent Owner object of this instance, default 0.
     */
    X509Extension(const QString &oid, const QString &oidShortName, const QString &oidLongName,
                  const QByteArray &value, bool critical = false, QObject *parent = 0);
    ~X509Extension();

    /**
     * Get object identifier.
     */
    const QString oid() const;
    /**
     * Get short name of object identifier.
     */
    const QString oidShortName() const;
    /**
     * Get long name of object identifier.
     */
    const QString oidLongName() const;

    /**
     * Get stored value.
     */
    const QByteArray &value() const;

    /** Get validity of extension. */
    bool isValid() const;

    /**
     * Get criticality flag.
     */
    bool isCritical() const;

    /**
     * Sets object identifier. String must be either a numerical OID,
     * a short or long name of an object identifier known to OpenSSL.
     *
     * @param OID Numerical, short or long name of a known OID.
     */
    void setOid(const QString &OID);
    /**
     * Sets object identifier with an object identifier unknown to OpenSSL.
     *
     * @param OID Numerical representation of OID.
     * @param longName Long name of OID.
     * @param shortName Short name of OID.
     */
    void setOid(const QString &OID, const QString &shortName, const QString &longName);

    /**
     * Set stored value.
     *
     * @param value New value to be stored.
     */
    void setValue(const QByteArray &value);

    /**
     * Set criticality flag.
     */
    void setCritical(bool critical);

    /** Assigns other to this instance. */
    X509Extension &operator=(const X509Extension &other);

  protected:
    /**
     * Create extension from OpenSSL's internal extension representation,
     * X509_EXTENSION.
     *
     * @param other X509_EXTENSION struct to be parsed.
     * @param parent Parent object of this instance, default 0. Not yet used.
     */
    X509Extension(X509_EXTENSION *other, QObject *parent = 0);

    /** Gets OpenSSL's internal name identifier. */
    int getNid() const;

    /**
     * Gets OpenSSL's internal representation of extension.
     */
    X509_EXTENSION *internal() const;

  private:
    int nid;
    bool critical, valid;
    QByteArray data;

    X509_EXTENSION *osslData;
};

};

#endif // _LIBKCAOSSL_X509EXTENSION_H
