/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _LIBKCAOSSL_UTILS_H
#define _LIBKCAOSSL_UTILS_H

#include "commons.h"

#include <openssl/asn1.h>
#include <openssl/evp.h>
#include <openssl/objects.h>
#include <openssl/x509.h>

#include <QtCore/QDateTime>
#include <QtCore/QList>
#include <QtCore/QString>

/** Namespace for kca-local classes */
namespace kCA_ossl {

class Certificate;
class Request;
class X509Extension;

/**
 * Contains a number of static helper functions to connect to
 * OpenSSL C API.
 *
 * @short OpenSSL C API helper functions
 * @author Felix Tiede <info@pc-tiede.de>
 */
class Utils
{
  public:
    /** Platform specific path to a source of random data */
    static const char *random_source;

    /**
     * Construct an ASN1_INTEGER from a quint64.
     *
     * @param in 64 bit integer from which to create the ASN1_INTEGER.
     * @return The input as ASN1_INTEGER if successful, otherwise 0.
     */
    static inline ASN1_INTEGER *convert(quint64 in)
    {
      ASN1_INTEGER *result = ASN1_INTEGER_new();
      if (result) {
        // Set serial number.
        BIGNUM *sn = BN_new();
        if (sn) {
          BN_hex2bn(&sn, QString().setNum(in, 16).toLocal8Bit().constData());
          BN_to_ASN1_INTEGER(sn, result);
          BN_free(sn);
        }
        else {
          ASN1_INTEGER_free(result);
          return 0;
        }
      }

      return result;
    }

    /** Convert an ASN1_BIT_STRING to a QByteArray. */
    static QByteArray print(const ASN1_BIT_STRING *in);
    /**
     * Construct a QDateTime from an ASN1_TIME struct.
     *
     * @param in ASN1_TIME struct to convert to QDateTime.
     * @return The given time as QDateTime if successful, otherwise invalid QDateTime.
     */
    static QDateTime convert(const ASN1_TIME *in);

    /**
     * Gets NID of an OID if it is already known to OpenSSL.
     *
     * @param oid Numerical representation, short or long name of object identifier.
     * @return OpenSSL's internal id if OID is known, NID_undef otherwise.
     */
    static inline int nid(const QString &oid)
    {
      return OBJ_txt2nid(oid.toLocal8Bit().constData());
    }

    /**
     * Gets object identifier of a NID.
     *
     * @param nid Internal identifier of object.
     * @return Object Identifier if nid is known in OpenSSL.
     */
    static QString oid(const int nid);

    /**
     * Populates a new OID in OpenSSL and assigns it as an alias of a
     * netscape comment for the runtime of this instance.
     *
     * @param oid Numerical representation of OID.
     * @param shortName short name of OID.
     * @param longName long name of OID.
     * @return @c -1 if OID is already registered but does not match @p shortName and @p longName.
     * @return @c 0 if unsuccessful.
     * @return @c \>0 if OID was successfully registered.
     */
    static int registerOid(const QString &oid, const QString &shortName, const QString &longName);

    /**
     * Returns specified digest for signing operation in OpenSSL.
     *
     * @param digest Digest to load, @see kCA::Digests
     * @return OpenSSL's internal representation of digest, default is SHA512
     */
    static inline const EVP_MD *load_digest(const Digest digest)
    {
      switch(digest) {
        case RIPEMD160:
          return EVP_ripemd160();
          break;
        case SHA1:
          return EVP_sha1();
          break;
        case SHA256:
          return EVP_sha256();
          break;
        case SHA384:
          return EVP_sha384();
          break;
        case SHA512:
        default:
          return EVP_sha512();
          break;
      }
    }

    /**
     * Populates given X509 structure with data for signing,
     * used by Certificate::signRequest() and Request::selfsign().
     *
     * @param[out] x X509 structure to be filled with data
     * @param[in] serial serial number of structure
     * @param[in] subject subject of structure
     * @param[in] issuer issuer of structure, may be equal to subject
     * @param[in] key public key of structure
     * @param[in] notBefore timestamp when validity begins
     * @param[in] notAfter timestamp whan validity ends
     * @return 0 if an error occurred and structure could not be populated, otherwise non-0
     */
    static int build_X509(X509 *x, quint64 serial, X509_NAME *subject, X509_NAME *issuer,
                          EVP_PKEY *key, const QDateTime &notBefore, const QDateTime &notAfter);

    /**
     * Turns integer constant reason into a string.
     *
     * @param reason integer revocation reason to convert to string
     * @return descriptive string of revocation reason
     */
    static inline const QString reasonToText(RevocationReason reason)
    {
      switch (reason) {
        case Unspecified:
          return QString("unspecified");
          break;
        case KeyCompromise:
          return QString("keyCompromise");
          break;
        case CACompromise:
          return QString("CACompromise");
          break;
        case AffilitionChanged:
          return QString("affiliationChanged");
          break;
        case Superseded:
          return QString("superseded");
          break;
        case CessationOfOperation:
          return QString("cessationOfOperation");
          break;
        case CertificateHold:
          return QString("certificateHold");
          break;
        case RemoveFromCRL:
          return QString("removeFromCRL");
          break;
        default:
          return QString("unspecified");
          break;
      }
    }

    /** Indicates if OpenSSL was already instructed to load decryption ciphers. */
    static bool OSSL_ciphers_loaded;

    /**
     * Standard extension list for S/MIME certificates.
     * Contains the following extensions and values (all non-critical):
     * <ul>
     * <li>basicConstraints=CA:FALSE
     * <li>keyUsage=dataEncipherment, digitalSignature, keyEncipherment
     * <li>extendedKeyUsage=emailProtection
     * <li>nsCertType=email
     * </ul>
     */
    static const QList< const X509Extension* > emailCertConstraints;
    /**
     * Standard extension list for CA certificates.
     * Contains the following extensions and values:
     * <ul>
     * <li>basicConstraints=CA:TRUE,critical
     * <li>keyUsage=cRLSign, keyCertSign
     * <li>nsCertType=sslCA, emailCA, objCA
     * </ul>
     */
    static const QList< const X509Extension* > CACertConstraints;

  private:
    /* Utils contains only static functions, thus there's no need for an instance. */
    Utils()  {}
    ~Utils() {}
};

}

#endif // _LIBKCAOSSL_UTILS_H
