/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _LIBKCAOSSL_REQUEST_H
#define _LIBKCAOSSL_REQUEST_H

#include "kca_ossl_export.h"

#include "commons.h"

#include <QtCore/QObject>
#include <QtCore/QDateTime>

class QFile;
template <typename T> class QList;
class QString;

struct X509_req_st;
typedef struct X509_req_st X509_REQ;

/** Namespace for kca-local classes */
namespace kCA_ossl
{

class Certificate;
class X509Name;
class X509Attribute;
class X509Extension;

class RequestPrivate;

/**
 * Class to manage signing requests.
 *
 * @short Certificate signing request
 * @author Felix Tiede <info@pc-tiede.de>
 */
class _LIBKCAOSSL_EXPORT Request : public QObject
{
  friend class RequestPrivate;
  friend class Certificate;

  Q_OBJECT
  public:
    /** Creates an empty request */
    Request(QObject *parent = 0);
    /** Copy constructor. */
    Request(const Request &other);
    /** Loads a request from a PEM or DER encoded file */
    Request(const QString &filename, QObject *parent = 0);
    ~Request();

    /**
     * Get state of request. Unless it was read from a file with a certificate
     * signing request or had its keypair newly generated, false is returned.
     */
    bool isValid() const;

    /** Get length of request's public key in number of bits. */
    int keyBits() const;

    /** Get subject of request. */
    const X509Name subject() const;

    /** Get list of additional attributes of request. */
    const QList<const X509Attribute*> &attributes() const;
    /** Get list of requested extensions for certificate to be generated. */
    const QList<const X509Extension*> &extensions() const;

    /**
     * Writes a request into a file. If a private key is in memory, it will also be written to a file.
     * Both files may be the same, but don't have to be. A passwort may be specified to protect the private
     * key using symmetric AES 256 CBC. If no password is given, the private key will not be protected.
     *
     * @param filename File to store the request.
     * @param mode Private key write mode, default is to not write private key.
     * @param keyfile File to store the private key, default is empty.
     * @param password password to protect private key, default is empty.
     */
    void toPemFile(const QString &filename, WriteMode mode = NoPrivateKey,
                   const QString &keyfile = QString(), const QString &password = QString()) const;

    /**
     * Writes private key (if any) to PEM encoded file, using symmetric AES 256 CBC to
     * with the given password to protect it.
     *
     * @param filename File to store the private key.
     * @param password Password to protect private key. if empty, key is stored as clear text.
     */
    void keyToPemFile(const QString &filename, const QString &password) const;

    /**
     * Create certificate from request by signing it with its own private key.
     * The own private key must be either provided by using generate() prior to
     * this function or loaded by giving keyFile and password to load it from file.
     * Aside from serial number and validity timestamps an arbitrary list of
     * X509Extension may be added to the created certificate. If no list was
     * specified, a default list will be added, based on if the subject's CN
     * ends with "ca" or "certification authority" in which case the used list
     * will be Utils::CACertConstraints. Otherwise Utils::emailCertConstraints
     * will be used.
     *
     * @param serial serial number of created certificate.
     * @param digest digest method to sign request.
     * @param notBefore timestamp at which validity timespan begins, default QDateTime::currentDateTime().
     * @param notAfter timestamp at which validity timespan ends, default QDateTime::currentDateTime() + 1 year.
     * @param extensions list of X509Extension to add to certificate, default empty list.
     * @param keyFile file containing private key for request, in DER or PEM format.
     * @param password password to protected private key, default is empty.
     * @return a certificate of request's public key.
     */
    const Certificate selfsign(quint64 serial, const Digest digest = SHA256,
                               const QDateTime &notBefore = QDateTime::currentDateTime(),
                               const QDateTime &notAfter = QDateTime::currentDateTime().addYears(1),
                               const QList< const X509Extension* > &extensions = QList< const X509Extension* >(),
                               const QString &keyFile = QString(), const QString &password = QString()) const;

    /**
     * Creates a new keypair of given length
     * for the given subject and with given attributes.
     *
     * @param subject Subject of request.
     * @param size length of key in bit, default 2048, less is considered unsafe for RSA keys and not allowed.
     * @param digest digest method to use to match private and public key.
     * @param attributes list of attributes to add to request, default empty list.
     */
    static Request generate(const X509Name &subject, int size = 2048, const Digest digest = SHA256,
                            const QList< const X509Attribute* > &attributes = QList< const X509Attribute* >());

  Q_SIGNALS:
    /**
     * Signalises progress of key generation.
     *
     * @param indicator character displaying type of progress made
     */
    void keyBuildProcess(const QChar &indicator);

  protected:
    /** Gets OpenSSL's internal representation of request. */
    X509_REQ *internal() const;

  private:
    /** Local value storage, protect binary compatibility. */
    RequestPrivate * const d;
};

}

#endif // _LIBKCAOSSL_REQUEST_H
