/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _LIBKCAOSSL_CERTIFICATE_H
#define _LIBKCAOSSL_CERTIFICATE_H

#include "kca_ossl_export.h"

#include "commons.h"

#include <QtCore/QObject>
#include <QtCore/QDateTime>

class QByteArray;
template <typename T> class QList;
class QSslCertificate;
class QString;

struct x509_st;
typedef struct x509_st X509;

/** Namespace for kca-local classes */
namespace kCA_ossl
{

class CertificatePrivate;
class Request;
class Revocationlist;
class X509Name;
class X509Extension;

/**
 * Class to manage certificates.
 *
 * @short Certificate manager
 * @author Felix Tiede <info@pc-tiede.de>
 */
class _LIBKCAOSSL_EXPORT Certificate : public QObject
{
  friend class CertificatePrivate;
  friend class Request;

  Q_OBJECT
  public:
    /** Creates an empty certificate */
    Certificate(QObject *parent = 0);
    /** Copy constructor */
    Certificate(const Certificate &other, QObject *parent = 0);

    /**
     * Creates a Certificate from DER or PEM encoded file.
     * A private key may be supplied in the same or another file.
     * A private key is required if this certificate is used to
     * sign requests. If the supplied private key is encrypted,
     * the decrypting password does not need to be supplied until the
     * private key is actually needed. The password is required each time
     * the private key is needed. The password is asked via password_cb,
     * thus password_cb should be given, if a private key is specified.
     *
     * @param filename DER or PEM encoded file from which to read certificate.
     * @param keyfile DER or PEM encoded file from which to read private key of certificate, default empty.
     * @param password password of private key of certificate, default null.
     * @param parent Owner of new instance.
     */
    Certificate(const QString &filename, const QString &keyfile = QString(),
                const QString &password = QString(), QObject *parent = 0);

    /**
     * Creates a certificate from a PEM-encoded byte array.
     * If a key is given, the protecting password if any is necessary as well.
     * @param certificate PEM encoded byte array from which to read certificate.
     * @param key PEM encoded byte array from which to read certificate's private key.
     * @param password Password to decrypt private key.
     * @param parent Owner of new instance.
     */
    Certificate(const QByteArray certificate,
                const QByteArray key = QByteArray(), const QString password = QString(), QObject* parent = 0);
    ~Certificate();

    /** Get state of certificate. Returns true if the constructor used was successful. */
    bool isValid() const;

    /** Get subject of certificate. */
    const X509Name subject() const;
    /** Get issuer subject of certificate. */
    const X509Name issuer() const;
    /** Get serial number of certificate. */
    quint64 serial() const;

    /** Get SHA1 fingerprint of certificate. */
    QByteArray fingerprint() const;

    /** Get hash value of certificate subject. */
    quint32 subjectHash() const;
    /** Get hash value of certificate issuer. */
    quint32 issuerHash() const;

    /** True, if certificate is a CA certificate, otherwise false. */
    bool isCA() const;

    /** Returns the key ID of certificate. */
    QString &keyId() const;

    /**
     * Returns the key ID of issuing certificate.
     * If the same as keyId() the certificate is selfsigned.
     */
    QString &issuerKeyId() const;

    /** Returns start timestamp of validity of certificate. */
    QDateTime &notBefore() const;

    /** Returns end timestamp of validity of certificate. */
    QDateTime &notAfter() const;

    /** Get digest used to sign certificate. */
    Digest digest() const;

    /** Get list of extensions of certificate. */
    const QList< const X509Extension* > &extensions() const;

    /**
     * Returns the certificate's private key in PEM format.
     * If the length of @param password is non-zero, the key is encrypted.
     * @param password Password to protect private key.
     * @return Private key in PEM format or empty if no private key available.
     */
    QByteArray key(const QString password = QString()) const;

    /**
     * Loads private key for certificate from file.
     *
     * @param filename PEM or DER encoded file to read key from.
     * @param password Phrase to decrypt key in file, default is empty.
     * @return true if private key could be loaded, otherwise false.
     */
    bool loadPrivateKey(const QString &filename, const QString &password = QString());

    /**
     * Add private key for certificate from raw data.
     *
     * @param key PEM encoded raw data array containing key.
     * @param password Phrase to decrypt key in raw data, default is empty.
     * @return true if private key could be loaded, otherwise false.
     */
    bool addPrivateKey(const QByteArray &key, const QString &password = QString());

    /** Unloads private key and frees occupied memory. */
    void unloadPrivateKey();

    /**
     * Signs request and returns a certificate with given timestamps and extensions.
     * If no extensions are specified, Utils::emailCertConstraints will be used.
     *
     * @param req Request to be signed with certificate's private key.
     * @param serial serial number of created certificate.
     * @param digest digest to use for signing request.
     * @param notBefore timestamp at which validity timespan begins, default QDateTime::currentDateTime().
     * @param notAfter timestamp at which validity timespan ends, default QDateTime::currentDateTime() + 1 year.
     * @param extensions list of X509Extension to add to certificate, default empty list.
     * @param parent owner of new created certificate.
     * @return 0 if either there is no private key to the certificate or the private key could not be decrypted, otherwise non-0
     */
    const Certificate sign(const Request &req, quint64 serial, const Digest digest = SHA256,
                           const QDateTime &notBefore = QDateTime::currentDateTime(),
                           const QDateTime &notAfter = QDateTime::currentDateTime().addYears(1),
                           const QList< const X509Extension* > &extensions = QList< const X509Extension* >(),
                           QObject *parent = 0) const;

    /**
     * Signs certificate revocation list with given digest.
     *
     * @param crl Certificate revocation list to be signed.
     * @param digest Digest to use for signing revocation list.
     * @param parent Owner of new created revocation list.
     */
    const Revocationlist sign(const Revocationlist &crl, const Digest digest = SHA256, QObject *parent = 0) const;

    /** Returns the ceretificate in raw PEM format. */
    QByteArray toPem() const;

    /**
     * Writes certificate to PEM encoded file.
     *
     * @param filename file to write information to.
     * @param mode Private key write mode, default is to not write private key.
     * @param keyfile File to store the private key, default is empty.
     * @param password password to protect private key, default is empty.
     * @return true if successful, otherwise false.
     */
    bool toPemFile(const QString &filename, WriteMode mode = NoPrivateKey,
                   const QString &keyfile = QString(), const QString &password = QString()) const;

    /**
     * Write certificate to DER encoded file. Private keys are not stored.
     *
     * @param filename File to write certificate to.
     * @return true if successful, otherwise false.
     */
    bool toDerFile(const QString &filename) const;

    /**
     * Export certificate to use directly within Qt.
     * Always returns a null pointer if libkca_ossl is compiled with -DWITH_QtNetwork=OFF.
     */
    const QSslCertificate *convert() const;

    /** Assignment operator. */
    Certificate& operator=(const Certificate &other);

    static Certificate fromRawData(const QByteArray &certificate,
                                   const QByteArray &key = QByteArray(), const QString &password = QString());

  protected:
    /**
     * Creates certificate from OpenSSL's internal certificate representation.
     *
     * @param other X509 struct to be parsed.
     * @param privateKey Private key of certificate, default 0.
     * @param parent Parent object of certificate, default 0.
     */
    Certificate(X509* other, const QByteArray &privateKey = QByteArray(), QObject* parent = 0);

    /** Gets OpenSSL's internal representation of certificate. */
    X509 *internal() const;

  private:
    /** Local value storage, protect binary compatibility. */
    CertificatePrivate * const d;
};

}

#endif // _LIBKCAOSSL_CERTIFICATE_H
