/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010, 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "../database.h"
#include "../backends/sqlite_db.h"

#include <QtTest/QTest>

#include <unistd.h>

using namespace kCA::Database;

class SQLiteTest: public QObject
{
  Q_OBJECT
  public:
    SQLiteTest(QObject* parent = 0) : QObject(parent), db(0) {}

  private slots:
    void initTestCase()
    {
      QCOMPARE(SQLiteDatabase::init("./kca_test.db"), true);
    }

    void testOpen()
    {
      db = new SQLiteDatabase("./kca_test.db");
      QCOMPARE(db->isOpen(), 0);
    }

    void testDefaultPolicy()
    {
      Database::Error error;
      QHash< QString, Policy > policies = db->getPolicies(&error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an eroor: %1. Test aborted.").arg(error).toUtf8().constData());
        return;
      }

      QCOMPARE(policies.count(), 1);
      QCOMPARE(policies.values().at(0).name, QString("default"));
    }

    void testInitExtensions()
    {
      Database::Error error;
      QList< Extension > extensions = db->getExtensions(&error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an eroor: %1. Test aborted.").arg(error).toUtf8().constData());
        return;
      }

      QCOMPARE(extensions.count(), 0);
    }

    void testAuthority()
    {
      Database::Error error;
      Authority testCA;

      // Database should be empty
      int authorities = db->getAuthorities(&error).count();
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getAuthorities() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QCOMPARE(authorities, 0);

      testCA = db->getAuthority(0UL, &error);
      QCOMPARE(error, Database::NotFound);

      testCA = db->getAuthority("non-existent", &error);
      QCOMPARE(error, Database::NotFound);

      // Prepare authority
      Policy policy;
      Extension extension;
      Authority authority;

      authority.name = "Test";
      authority.hash = 1UL;
      authority.certificateDays = 360;
      authority.crlnumber = 0;
      authority.crldays = 4;
      authority.digest = kCA_ossl::SHA384;

      policy.name = "Match";
      policy.country = Policy::Match;
      policy.state = Policy::Match;
      policy.location = Policy::Match;
      policy.organization = Policy::Match;
      policy.organizationalUnit = Policy::Match;
      policy.commonName = Policy::Supplied;
      policy.email = Policy::Supplied;
      authority.policy = policy;

      extension.id = 0;
      extension.type = Extension::Certificate;
      extension.oid = "OBJ_basic_constraints";
      extension.critical = true;
      extension.value = QByteArray("CA:true");
      authority.extensions << extension;

      authority.certificate = QByteArray("Not so random cert data");
      authority.key = QByteArray("Not so random key data");

      error = db->setAuthority(authority);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. setAuthority() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      authorities = db->getAuthorities(&error).count();
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getAuthorities() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QVERIFY(authorities > 0);

      // Test persistence over DB close/open.
      delete db;
      db = new SQLiteDatabase("./kca_test.db");
      if (!db->isOpen()) {
        QWARN(QString("Could not reopen database. Test aborted.").toUtf8().constData());
        return;
      }

      // Requesting authority by hash value
      testCA = db->getAuthority(authority.hash, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getAuthority(ulong) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QCOMPARE(testCA.keyDirty, false);
      QVERIFY(testCA.name == authority.name);
      QVERIFY(testCA.certificateDays == authority.certificateDays);
      QVERIFY(testCA.crlnumber == authority.crlnumber);
      QVERIFY(testCA.crldays == authority.crldays);
      QVERIFY(testCA.digest == authority.digest);
      QVERIFY(testCA.certificate == authority.certificate);
      QVERIFY(testCA.key == authority.key);
      QVERIFY(testCA.policy.name == policy.name);
      QVERIFY(testCA.policy.country == policy.country && testCA.policy.state == policy.state
              && testCA.policy.location == policy.location && testCA.policy.organization == policy.organization
              && testCA.policy.organizationalUnit == policy.organizationalUnit
              && testCA.policy.commonName == policy.commonName && testCA.policy.email == policy.email);
      QCOMPARE(testCA.extensions.count(), 1);
      QVERIFY(testCA.extensions.at(0).type == extension.type
              && testCA.extensions.at(0).oid == extension.oid
              && testCA.extensions.at(0).critical == extension.critical
              && testCA.extensions.at(0).value == extension.value);

      // Requesting Authority by case-insensitive name
      testCA = db->getAuthority(authority.name.toUpper(), &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getAuthority(QString) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QCOMPARE(testCA.keyDirty, false);
      QVERIFY(testCA.name == authority.name);
      QVERIFY(testCA.certificateDays == authority.certificateDays);
      QVERIFY(testCA.crlnumber == authority.crlnumber);
      QVERIFY(testCA.crldays == authority.crldays);
      QVERIFY(testCA.digest == authority.digest);
      QVERIFY(testCA.certificate == authority.certificate);
      QVERIFY(testCA.key == authority.key);
      QVERIFY(testCA.policy.name == policy.name);
      QVERIFY(testCA.policy.country == policy.country && testCA.policy.state == policy.state
              && testCA.policy.location == policy.location && testCA.policy.organization == policy.organization
              && testCA.policy.organizationalUnit == policy.organizationalUnit
              && testCA.policy.commonName == policy.commonName && testCA.policy.email == policy.email);
      QCOMPARE(testCA.extensions.count(), 1);
      QVERIFY(testCA.extensions.at(0).type == extension.type
              && testCA.extensions.at(0).oid == extension.oid
              && testCA.extensions.at(0).critical == extension.critical
              && testCA.extensions.at(0).value == extension.value);

      // These should still not exist in the database
      testCA = db->getAuthority(0UL, &error);
      QCOMPARE(error, Database::NotFound);

      testCA = db->getAuthority("non-existent", &error);
      QCOMPARE(error, Database::NotFound);
    }

    void testPostAuthorityPolicy()
    {
      Database::Error error;
      QHash< QString, Policy > policies = db->getPolicies(&error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an eroor: %1. Test aborted.").arg(error).toUtf8().constData());
        return;
      }

      QVERIFY(policies.count() != 0);
      QCOMPARE(policies.values().at(0).name, QString("default"));
    }

    void testPostAuthorityExtensions()
    {
      Database::Error error;
      QList< Extension > extensions = db->getExtensions(&error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an eroor: %1. Test aborted.").arg(error).toUtf8().constData());
        return;
      }

      QVERIFY(extensions.count() != 0);
    }

    void testUpdateAuthority()
    {
      Database::Error error;
      Authority ca = db->getAuthorities(&error).values().at(0);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update authority test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      ca.name = "Testchanged name";
      ++ca.crlnumber;
      ca.digest = kCA_ossl::RIPEMD160;
      ca.certificate = "Not really random test-changed certificate data";
      ca.key = "New keypath";
      ca.keyDirty = true;
      error = db->setAuthority(ca);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. setAuthority() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      Authority testCA = db->getAuthorities(&error).values().at(0);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update authority comparison test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      QCOMPARE(testCA.hash, ca.hash);
      QCOMPARE(testCA.name, ca.name);
      QCOMPARE(testCA.crlnumber, ca.crlnumber);
      QVERIFY(testCA.digest == ca.digest);
      QVERIFY(testCA.certificate != ca.certificate);
      QVERIFY(testCA.key == ca.key);
    }

    void testUpdateAuthorityCert()
    {
      Database::Error error;
      quint32 hash = db->getAuthorities(&error).values().at(0).hash;
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update authority test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      error = db->setAuthorityCertificate(0, QByteArray());
      QCOMPARE(error, Database::ConstraintError);

      QByteArray testCert = "Certificate update test data";
      QByteArray testKey = "Certificate update test key data";
      error = db->setAuthorityCertificate(0, testCert);
      QCOMPARE(error, Database::NotFound);

      error = db->setAuthorityCertificate(hash, testCert, testKey);
      QCOMPARE(error, Database::NoError);

      Authority testCA = db->getAuthority(hash, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update authority test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      QCOMPARE(testCA.hash, hash);
      QVERIFY(testCA.certificate == testCert);
      QVERIFY(testCA.key == testKey);
    }

    void testCertificate()
    {
      Database::Error error;
      Certificate testCert;

      // Database should contain no certificates
      int certificates = db->getCertificates(0UL, &error).count();
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getCertificates(0) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QCOMPARE(certificates, 0);

      testCert = db->getCertificate(1UL, 1, &error);
      QCOMPARE(error, Database::NotFound);

      Certificate cert;
      cert.authorityhash = 0UL;
      cert.fingerprint = QByteArray().setNum(2, 16);
      cert.owner = "Unittest User";
      cert.certificate = "Not so random certificate data";
      cert.issued = QDateTime::currentDateTime().addDays(1);
      cert.expires = QDateTime::currentDateTime();
      cert.serial = 5UL;
      cert.state = Certificate::Valid;
      cert.subject = "CN=Unittest Certificate";

      error = db->setCertificate(cert);
      QCOMPARE(error, Database::ConstraintError);

      if (error != Database::NoError && error != Database::ConstraintError) {
        QWARN(QString("Database reported an error: %1. setCertificate() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      // Should be still none certificate, constraint error.
      certificates = db->getCertificates(0UL, &error).count();
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getCertificates(0) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QCOMPARE(certificates, 0);

      // Set authority to an existent authority to match constraint. Dates still fail.
      cert.authorityhash = 1UL;
      error = db->setCertificate(cert);
      QCOMPARE(error, Database::ConstraintError);

      if (error != Database::NoError && error != Database::ConstraintError) {
        QWARN(QString("Database reported an error: %1. setCertificate() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      cert.issued = QDateTime::currentDateTime();
      cert.expires = QDateTime::currentDateTime().addSecs(1);

      error = db->setCertificate(cert);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getCertificates(0) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      // Insert second cert for later use with longer runtime
      cert.fingerprint = QByteArray().setNum(3, 16);
      cert.serial = 6UL;
      cert.expires = QDateTime::currentDateTime().addSecs(3600);
      db->setCertificate(cert);

      testCert = db->getCertificate("2", &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getCertificate(fingerprint) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QVERIFY(testCert.fingerprint == QByteArray().setNum(2, 16));

      // Requesting authority by hash value
      testCert = db->getCertificate(cert.authorityhash, cert.serial, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getCertificate(ulong, quint64) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      QVERIFY(testCert.owner == cert.owner);
      QVERIFY(testCert.authorityhash == cert.authorityhash);
      QVERIFY(testCert.certificate == cert.certificate);
      QVERIFY(testCert.issued.secsTo(cert.issued) == 0);
      QVERIFY(testCert.expires.secsTo(cert.expires) == 0);
      QVERIFY(testCert.serial == cert.serial);
      QVERIFY(testCert.state == cert.state);
      QVERIFY(testCert.subject == cert.subject);

      testCert = db->getCertificate(1UL, cert.serial, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. getCertificate(ulong, uint64) test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      QVERIFY(testCert.owner == cert.owner);
      QVERIFY(testCert.authorityhash == cert.authorityhash);
      QVERIFY(testCert.certificate == cert.certificate);
      QVERIFY(testCert.issued.secsTo(cert.issued) == 0);
      QVERIFY(testCert.expires.secsTo(cert.expires) == 0);
      QVERIFY(testCert.fingerprint == cert.fingerprint);
      QVERIFY(testCert.state == cert.state);
      QVERIFY(testCert.subject == cert.subject);

      cert.serial = 7UL;
      cert.fingerprint = QByteArray().setNum(4, 16);
      cert.issued = QDateTime::currentDateTime();
      cert.expires = QDateTime::currentDateTime().addSecs(1);
      db->setCertificate(cert);

      QTest::qWait(3000);
      cert.serial = 8UL;
      cert.fingerprint = QByteArray().setNum(5, 16);
      db->setCertificate(cert);

      testCert = db->getCertificate(1UL, 7UL, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. auto-expiration test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      QVERIFY(testCert.state == Certificate::Expired);
    }

    void testUpdateCertificate()
    {
      Database::Error error;
      Certificate cert = db->getCertificate(1UL, 6UL, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QByteArray certificate = cert.certificate;

      cert.owner = "New test owner";
      cert.certificate = "New not so random cert data.";
      cert.state = Certificate::Revoked;
      cert.revoked = QDateTime::currentDateTime().addSecs(100);
      cert.reason = kCA_ossl::CertificateHold;

      error = db->setCertificate(cert);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. setCertificate() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      Certificate testCert = db->getCertificate(1UL, 6UL, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update certificate test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      QVERIFY(testCert.owner == cert.owner);
      QVERIFY(testCert.certificate != cert.certificate);
      QVERIFY(testCert.certificate == certificate);
      QVERIFY(testCert.reason == cert.reason);

      cert = db->getCertificate(1UL, 6UL, &error);
      cert.state = Certificate::Valid;
      error = db->setCertificate(cert);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. setCertificate() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      testCert = db->getCertificate(1UL, 6UL, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update certificate test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QVERIFY(testCert.state == Certificate::Valid);

      cert.reason = kCA_ossl::CessationOfOperation;
      cert.revoked = QDateTime::currentDateTime();
      cert.state = Certificate::Revoked;
      error = db->setCertificate(cert);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. setCertificate() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      testCert = db->getCertificate(1UL, 6UL, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update certificate test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QVERIFY(testCert.state == Certificate::Revoked);

      cert.state = Certificate::Valid;
      error = db->setCertificate(cert);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. setCertificate() test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }

      testCert = db->getCertificate(1UL, 6UL, &error);
      if (error != Database::NoError) {
        QWARN(QString("Database reported an error: %1. Update certificate test aborted.").arg(error).toUtf8().constData());
        QWARN(QString("Backend message: '%1'").arg(db->lastError()).toUtf8().constData());
        return;
      }
      QVERIFY(testCert.state != Certificate::Valid);
    }

    void testSerial()
    {
      Database::Error error;
      QCOMPARE(db->isSerialUsed(1UL, 2UL, &error), false);
      QCOMPARE(db->isSerialUsed(1UL, 5UL, &error), true);
    }

    void cleanupTestCase()
    {
      if (db)
        delete db;
      unlink("./kca_test.db");
    }

  private:
    Database *db;
};

QTEST_MAIN(SQLiteTest)
#include "sqlite_db_test.moc"
