/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010, 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "../authority.h"

#include "../database.h"

#ifdef USE_SQLITE3
#include "../backends/sqlite_db.h"
#define DB_BACKEND Database::SQLiteDatabase
#endif

#include <libkca_ossl/certificate.h>
#include <libkca_ossl/commons.h>
#include <libkca_ossl/request.h>
#include <libkca_ossl/revocationlist.h>
#include <libkca_ossl/x509extension.h>
#include <libkca_ossl/x509name.h>

#include <QtTest/QTest>

#include <unistd.h>

using namespace kCA;

class AuthorityTest : public QObject
{
  Q_OBJECT
  public:
    AuthorityTest(QObject* parent = 0) : QObject(parent), db(0), _authority(0) {}

  private slots:
    void initTestCase()
    {
      QCOMPARE(DB_BACKEND::init("./authority_test.db"), true);
      db = new DB_BACKEND("./authority_test.db");
      QCOMPARE(db->isOpen(), 0);
    }

    void newAuthority()
    {
      kCA_ossl::Certificate cert;
      _authority = new Authority(db, "Testauthority", db->getPolicies().values().at(0), cert, "");
      QVERIFY(_authority->name() == "Testauthority");
      QVERIFY(_authority->getPolicy().name == "default");
    }

    void testSetName()
    {
      QVERIFY(_authority->name() == "Testauthority");
      _authority->setName("newNamedTestauthority");
      QVERIFY(_authority->name() == "newNamedTestauthority");
    }

    void testSetTimes()
    {
      unsigned int current;
      current = _authority->certificateDays();
      _authority->setCertificateDays(current+1);
      QVERIFY(_authority->certificateDays() != current);
      QVERIFY(_authority->certificateDays() == (current+1));

      current = _authority->crlDays();
      _authority->setCrlDays(current+1);
      QVERIFY(_authority->crlDays() != current);
      QVERIFY(_authority->crlDays() == (current+1));
    }

    void testSetDigest()
    {
      kCA_ossl::Digest current;
      current = _authority->digest();
      QVERIFY(current == kCA_ossl::SHA1);

      _authority->setDigest(kCA_ossl::SHA512);
      QVERIFY(_authority->digest() == kCA_ossl::SHA512);
      QVERIFY(_authority->digest() != current);
    }

    void testCrlDistributionPoint()
    {
      QUrl dp("http://example.com/test.crl");
      QVERIFY(_authority->crlDistributionPoint().isEmpty());
      _authority->setCrlDistributionPoint(dp);
      QVERIFY(_authority->crlDistributionPoint() == dp);
      QUrl dp2("http://www.example.com/test2.crl");
      _authority->setCrlDistributionPoint(dp2);
      QVERIFY(_authority->crlDistributionPoint() != dp2);
      QVERIFY(_authority->crlDistributionPoint() == dp);
    }

    void testSetPolicy()
    {
      Database::Policy inPolicy, outPolicy;
      inPolicy.name = "Provided";
      inPolicy.country = Database::Policy::Supplied;
      inPolicy.state = Database::Policy::Supplied;
      inPolicy.location = Database::Policy::Supplied;
      inPolicy.organization = Database::Policy::Supplied;
      inPolicy.organizationalUnit = Database::Policy::Optional;
      inPolicy.commonName = Database::Policy::Supplied;
      inPolicy.email = Database::Policy::Supplied;

      _authority->setPolicy(inPolicy);
      outPolicy = _authority->getPolicy();
      QCOMPARE(inPolicy.name, outPolicy.name);
      QVERIFY(inPolicy.state == outPolicy.state);
      QVERIFY(inPolicy.organizationalUnit != outPolicy.organization);
      QVERIFY(inPolicy.organizationalUnit == outPolicy.organizationalUnit);
    }

    void testSetExtensions()
    {
      _authority->setCertificateExtensions(kCA_ossl::emailCertExtensions());
      QVERIFY(_authority->certificateExtensions() == kCA_ossl::emailCertExtensions());
      QVERIFY(_authority->crlExtensions() != kCA_ossl::emailCertExtensions());

      _authority->setCrlExtensions(kCA_ossl::emailCertExtensions());
      QVERIFY(_authority->crlExtensions() == kCA_ossl::emailCertExtensions());
    }

    void testWorkingAuthority()
    {
      QString name = "/C=DE/L=Hamburg/ST=Hamburg/O=Felix Tiede/CN=Unit Test CA/email=info@pc-tiede.de/";
      kCA_ossl::Request req = kCA_ossl::Request::generate(kCA_ossl::X509Name(name), 512);
      kCA_ossl::Certificate cert = req.selfsign(kCA_ossl::generateRandom(), kCA_ossl::SHA256,
                                                QDateTime::currentDateTime(), QDateTime::currentDateTime().addSecs(3600));

      Authority auth(db, "Testing authority", db->getPolicies().value("default"), cert, cert.key("TestKey Passphrase"));
      QVERIFY(db->getAuthorities().count() > 1);
      QVERIFY(auth.hash() == cert.subjectHash());
      QVERIFY(auth.issuerHash() == cert.issuerHash());

      auth.setCertificateExtensions(kCA_ossl::emailCertExtensions());
      QVERIFY(auth.certificateExtensions() == kCA_ossl::emailCertExtensions());

      if (_authority)
        delete _authority;

      _authority = new Authority(db, auth.hash());
      QCOMPARE(_authority->hash(), auth.hash());
      QVERIFY(_authority->certificateExtensions().count() == 0);
    }

    void testKeyPath()
    {
      QCOMPARE(_authority->keyPath().isEmpty(), true);
    }

    void testUnlockKey()
    {
      QString name = "/C=DE/L=Hamburg/ST=Hamburg/O=Felix Tiede/CN=Unit Test key/email=info@pc-tiede.de/";
      kCA_ossl::Request falseKey = kCA_ossl::Request::generate(kCA_ossl::X509Name(name), 768);
      falseKey.keyToPemFile("./test.key", "FalseKey Passphrase");

      QCOMPARE(_authority->unlockKey("FalseKey Passphrase", false, "./test.key"), false);
      unlink("./test.key");

      QCOMPARE(_authority->keyUnlocked(), false);
      QCOMPARE(_authority->unlockKey("TestKey Passphrase"), true);
      QCOMPARE(_authority->keyUnlocked(), true);
      _authority->lockKey();
      QCOMPARE(_authority->keyUnlocked(), false);
    }

    void testCertSigning()
    {
      connect(_authority, SIGNAL(requestSigned(kCA_ossl::Certificate, int)), this, SLOT(testCert(kCA_ossl::Certificate, int)));

      QString certname = "/C=DE/L=Hamburg/ST=Hamburg/O=Felix Tiede/CN=Unit Test Certificate/email=info@pc-tiede.de/";
      kCA_ossl::Request req = kCA_ossl::Request::generate(kCA_ossl::X509Name(certname));

      if (!_authority->keyUnlocked())
        QCOMPARE(_authority->unlockKey("TestKey Passphrase", true), true);

      _authority->signRequest("Test owner", req,
                              QDateTime::currentDateTime(), QDateTime::currentDateTime().addSecs(120));
      QCOMPARE(_authority->keyUnlocked(), true);
      _authority->lockKey();
    }

    void testRevoke()
    {
      connect(_authority, SIGNAL(certificateRevoked(quint32, quint64, int, int)), SLOT(testRevoked(quint32, quint64, int, int)));

      if (!_authority->keyUnlocked())
        QCOMPARE(_authority->unlockKey("TestKey Passphrase"), true);

      _authority->revokeCertificate(certserial, kCA_ossl::CessationOfOperation, QDateTime());
      QCOMPARE(_authority->keyUnlocked(), false);
    }

    void testCrlCreation()
    {
      QTest::qWait(1000);

      connect(_authority, SIGNAL(crlGenerated(kCA_ossl::Revocationlist, int)), this, SLOT(testCRL(kCA_ossl::Revocationlist, int)));

      if (!_authority->keyUnlocked())
        QCOMPARE(_authority->unlockKey("TestKey Passphrase"), true);

      QCOMPARE(_authority->keyUnlocked(), true);
      _authority->generateCrl(kCA_ossl::SHA384);
      QCOMPARE(_authority->keyUnlocked(), false);
    }

    void testReplaceCertificate()
    {
      QString name = "/C=DE/L=Hamburg/ST=Hamburg/O=Felix Tiede/CN=Unit Test replacement certificate/email=info@pc-tiede.de/";
      kCA_ossl::Request req = kCA_ossl::Request::generate(kCA_ossl::X509Name(name), 384);
      kCA_ossl::Certificate replacement = req.selfsign(kCA_ossl::generateRandom(), kCA_ossl::SHA512,
                                                       QDateTime::currentDateTime(), QDateTime::currentDateTime().addSecs(3600));
      req.keyToPemFile("./replacement_test.key", "Replacement Passphrase");
      req.keyToPemFile("./replacement_test2.key", "");
      replacement.unloadPrivateKey();

      name = "/C=DE/L=Hamburg/ST=Hamburg/O=Felix Tiede/CN=Unit Test CA/email=info@pc-tiede.de/";
      kCA_ossl::Request req_work = kCA_ossl::Request::generate(kCA_ossl::X509Name(name), 384);
      kCA_ossl::Certificate replacement_work = req_work.selfsign(kCA_ossl::generateRandom(), kCA_ossl::SHA512,
                                                                 QDateTime::currentDateTime(),
                                                                 QDateTime::currentDateTime().addSecs(3600));
      req_work.keyToPemFile("./replacement_work.key", "Replacement Passphrase");
      req_work.keyToPemFile("./replacement_work2.key", "");
      replacement_work.unloadPrivateKey();

      QCOMPARE(_authority->replaceCertificate(replacement, QByteArray(), "Replacement Passphrase"), false);

      if (!_authority->keyUnlocked())
        QCOMPARE(_authority->unlockKey("TestKey Passphrase", true), true);

      QCOMPARE(_authority->replaceCertificate(replacement, "Replacement testkey"), false);
      QCOMPARE(_authority->replaceCertificate(replacement, QByteArray(), "Replacement"), false);
      QCOMPARE(_authority->keyUnlocked(), true);
      QCOMPARE(_authority->replaceCertificate(replacement, "./replacement_test.key"), false);
      QCOMPARE(_authority->replaceCertificate(replacement, "./replacement_test.key", "Replacement Passphrase"), false);
      unlink("./replacement_test.key");
      unlink("./replacement_test2.key");

      QCOMPARE(_authority->keyUnlocked(), true);
      QCOMPARE(_authority->replaceCertificate(replacement_work, "./replacement_work.key"), false);
      QCOMPARE(_authority->replaceCertificate(replacement_work, "./replacement_work.key", "Replacement Passphrase"), true);

      QCOMPARE(_authority->unlockKey("", false, "replacement_work2.key"), true);

      _authority->lockKey();

      unlink("./replacement_work.key");
      unlink("./replacement_work2.key");
    }

    void cleanupTestCase()
    {
      if (db)
        delete db;
      unlink("./authority_test.db");
    }

    // Slots for Authority signals
    void testCert(const kCA_ossl::Certificate& cert, int error)
    {
      if (error == Authority::DatabaseError) {
        QWARN("Database error on request signing.");
        kCA::Authority* caller =(kCA::Authority*) sender();
        if (caller)
          QWARN(QString("Database backend error message: %1").arg(caller->databaseError()).toUtf8().constData());
        return;
      }

      QVERIFY(error == Authority::NoError);
      QCOMPARE(cert.issuerHash(), _authority->hash());
      QVERIFY((certserial = cert.serial()) > 0);
    }

    void testRevoked(const quint32 authorityHash, const quint64 serial, const int reason, const int error)
    {
      QCOMPARE(authorityHash, _authority->hash());
      QCOMPARE(serial, certserial);
      QCOMPARE(reason,(int) kCA_ossl::CessationOfOperation);
      if (error == Authority::DatabaseError) {
        QWARN("Database error on certificate revoking.");
        kCA::Authority* caller =(kCA::Authority*) sender();
        if (caller)
          QWARN(QString("Database backend error message: %1").arg(caller->databaseError()).toUtf8().constData());
        return;
      }
    }

    void testCRL(const kCA_ossl::Revocationlist& crl, int error)
    {
      if (error == Authority::DatabaseError) {
        QWARN("Database error on CRL generation.");
        kCA::Authority* caller =(kCA::Authority*) sender();
        if (caller)
          QWARN(QString("Database backend error message: %1").arg(caller->databaseError()).toUtf8().constData());
        return;
      }

      QVERIFY(error == Authority::NoError);
      QVERIFY(crl.extensions().count() > 0);
      foreach(const kCA_ossl::X509Extension* ext, crl.extensions())
        if (ext->oid() == "2.5.29.20")
          QVERIFY(ext->value().toInt() > 0);

      QVERIFY(crl.entries().count() > 0);
    }

  private:
    Database::Database* db;
    Authority* _authority;

    quint64 certserial;
};

QTEST_MAIN(AuthorityTest);
#include "authority_test.moc"
