/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "signrequestdialog.h"

#include "authority.h"

#include <libkca_ossl/request.h>
#include <libkca_ossl/x509extension.h>
#include <libkca_ossl/x509name.h>

#include <QtCore/QDate>
#include <QtCore/QDateTime>
#include <QtCore/QMap>
#include <QtCore/QString>
#include <QtCore/QTime>

#include <QtGui/QListWidgetItem>
#include <QtGui/QMessageBox>
#include <QtGui/QPushButton>

#include <kdatetimewidget.h>

SignRequestDialog::SignRequestDialog(const QHash< quint32, kCA::Authority* >& authorityList,
                                     const kCA_ossl::Request& request, Settings* settings,
                                     QWidget* parent, Qt::WFlags flags) :
  QDialog(parent, flags),
  authorityUnlocked(false),
  settings(settings),
  authorityList(authorityList),
  requestData(request.subject())
{
  Q_ASSERT(this->settings);
  setupUi(this);

  buttonBox->button(QDialogButtonBox::Yes)->setAutoDefault(false);
  buttonBox->button(QDialogButtonBox::Yes)->setDefault(false);
  buttonBox->button(QDialogButtonBox::No)->setAutoDefault(true);
  buttonBox->button(QDialogButtonBox::No)->setDefault(true);

  QString requestText = request.subject().toString();
  requestText.remove(0, 1);
  requestText.replace('/', '\n');
  requestText.replace('=', ":\t");

  if (requestText.endsWith('\n'))
    requestText.remove(requestText.lastIndexOf('\n'), 1);

  requestName->setPlainText(requestText);

  startDate->setDate(QDate::currentDate());
  startTime->setTime(QTime::currentTime());
  stopDate->setDate(QDate::currentDate());
  stopTime->setTime(QTime::currentTime());

  cbSupersede->setChecked(settings->supersede);

  // Display all requested extension for selection by user
  int i = 0;
  QString extDescription;
  QListWidgetItem *entry;
  foreach(const kCA_ossl::X509Extension* ext, request.extensions()) {
    extDescription = QString("%1: %2 = %3").arg(++i).arg(ext->oidLongName()).arg(QString(ext->value()));
    extMap.insert(extDescription, ext);
    entry = new QListWidgetItem(extDescription);
    entry->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled);
    entry->setCheckState(Qt::Checked);
    extensionsList->addItem(entry);
  }

  // Populate authority combobox
  foreach(quint32 hash, authorityList.keys())
    authorities->addItem(authorityList.value(hash)->name(), hash);

  connect(authorities, SIGNAL(currentIndexChanged(int)), SLOT(authoritySelected(int)));

  connect(startDate, SIGNAL(dateChanged(QDate)), SLOT(dateChanged(QDate)));
  connect(startTime, SIGNAL(timeChanged(QTime)), SLOT(timeChanged(QTime)));
  connect(stopDate, SIGNAL(dateChanged(QDate)), SLOT(dateChanged(QDate)));
  connect(stopTime, SIGNAL(timeChanged(QTime)), SLOT(timeChanged(QTime)));

  connect(cbSupersede, SIGNAL(clicked(bool)), SLOT(supersedeChanged(bool)));

  connect(buttonBox, SIGNAL(accepted()), SLOT(accept()));
  connect(buttonBox, SIGNAL(rejected()), SLOT(reject()));

  // Set default values _after_ connecting signals to produce warnings
  authorities->setCurrentIndex(authorities->findData(settings->authority));
  startDate->setDate(settings->notBefore.date());
  startTime->setTime(settings->notBefore.time());
  stopDate->setDate(settings->notAfter.date());
  stopTime->setTime(settings->notAfter.time());
}

SignRequestDialog::~SignRequestDialog()
{

}

void SignRequestDialog::accept()
{
  // Populate Settings::extensions with selected extensions
  QListWidgetItem *entry;
  for (int i = 0; i < extensionsList->count(); ++i) {
    entry = extensionsList->item(i);
    if (entry && entry->checkState() == Qt::Checked)
      settings->extensions << extMap.value(entry->text());
  }

  this->done(QDialog::Accepted);
}

void SignRequestDialog::reject()
{
  this->done(QDialog::Rejected);
}

void SignRequestDialog::authoritySelected(const int index)
{
  settings->authority = authorities->itemData(index).toULongLong();
  kCA::Authority *authority = authorityList.value(settings->authority, 0);
  if (authority) {
    if (!authority->enabled()) {
      authorityUnlocked = false;
      buttonBox->button(QDialogButtonBox::Yes)->setEnabled(false);
      QMessageBox::critical(this, i18n("Authority unavailable"),
                            i18n("The selected authority is out of validity period or was revoked."), QMessageBox::Ok);
      return;
    }
    if (!requestData.keys().isEmpty() && !authority->matchPolicy(requestData)) {
      authorityUnlocked = false;
      buttonBox->button(QDialogButtonBox::Yes)->setEnabled(false);
      QMessageBox::critical(this, tr("Authority policy mismatch"),
                            tr("The request's subject is not accepted by the current authority."), QMessageBox::Ok);
      return;
    }
    stopDate->setDateTime(startDate->dateTime().addDays(authority->certificateDays()));
    authorityUnlocked = true;
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(true);
  }
}

void SignRequestDialog::dateChanged(const QDate& value)
{
  if (value < QDate::currentDate()) {
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(false);
    return;
  }
  else
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(authorityUnlocked);

  if (stopDate->date() < startDate->date() ||
    (stopDate->date() == startDate->date() && stopTime->time() < startTime->time())) {
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(false);
    return;
  }
  else
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(authorityUnlocked);

  if (sender()->objectName() == "startDate") {
    settings->notBefore.setDate(value);
    if (authorityList.contains(settings->authority))
      stopDate->setDate(value.addDays(authorityList[settings->authority]->certificateDays()));
  }
  if (sender()->objectName() == "stopDate")
    settings->notAfter.setDate(value);
}

void SignRequestDialog::timeChanged(const QTime& value)
{
  if (value < QTime::currentTime().addSecs(60)) {
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(false);
    return;
  }
  else
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(authorityUnlocked);

  if (stopDate->date() == startDate->date() && stopTime->time() < startTime->time()) {
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(false);
    return;
  }
  else
    buttonBox->button(QDialogButtonBox::Yes)->setEnabled(authorityUnlocked);

  if (sender()->objectName() == "startTime") {
    settings->notBefore.setTime(value);
  }
  if (sender()->objectName() == "stopTime")
    settings->notAfter.setTime(value);
}

void SignRequestDialog::supersedeChanged(const bool value)
{
  settings->supersede = value;
}
