/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2007 - 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef KCAVIEW_H
#define KCAVIEW_H

#include <QtCore/QHash>
#include <QtGui/QWidget>

#include "ui_kcaview_base.h"

template< typename T > class QList;
class QStringList;

class QPainter;
class QTreeWidgetItem;

class KUrl;

/**
 * This is the main view class for kca.  Most of the non-menu,
 * non-toolbar, and non-statusbar (e.g., non frame) GUI code should go
 * here.
 *
 * @short Main view
 * @author Felix Tiede <info@pc-tiede.de>
 * @version %{VERSION}
 */

class kcaView : public QWidget, public Ui::kcaview_base
{
    Q_OBJECT
public:
    /**
     * Default constructor
     */
    kcaView(QWidget *parent);

    /**
     * Destructor
     */
    virtual ~kcaView();

    /** Get subject hash value of currently selected authority. */
    quint32 activeAuthority() const;
    /** Get serial number of currently selected certificate. */
    quint64 activeCertificate() const;

    /**
     * Add an authority to the main tree.
     * @param parent Subject hash value of parent authority if any, if identical to hash,
     * the added authority is added as a root element.
     * @param hash Subject hash value of added authority, used to identify which authority is selected, @see activeAuthority().
     * @param name Descriptive name shown in the tree.
     * @param unlocked Locking state of added authority.
     * @param lockable Lockability of added authority, should be false for every authority
     * which has its key stored unencrypted in the database. defaults to false. Authorities with
     * this flag set to true can't be locked by GUI later.
     */
    void addAuthority(const quint32 parent, const quint32 hash, const QString& name,
                      const bool unlocked = false, const bool lockable = true);
    void clearAuthorities();

    /** Display certificates in list. */
    void showCertificates(const QList< QStringList >& certificates);
    /** Add new certificate to list. */
    void addCertificate(const QStringList& certificate);

public slots:
    void authorityLocked(const quint32 authority);
    void authorityUnlocked(const quint32 authority);

signals:
    void authoritySelected(const quint32 authority);
    void certificateSelected(const quint32 authority, const quint64 certificate);
    void generateAllCrls();
    void generateCrl();

    /**
     * Use this signal to change the content of the statusbar
     */
    void signalChangeStatusbar(const QString& text);

    /**
     * Use this signal to change the content of the caption
     */
    void signalChangeCaption(const QString& text);

private slots:
    void authoritySelected(QTreeWidgetItem* item);
    void certificateSelected(QTreeWidgetItem* item);

    void switchColors();
    void settingsChanged();

private:
    /** Collection of flags controlling GUI behaviour based on which authority is selected. */
    struct AuthorityFlags {
      bool lockable;
      bool unlocked;
    };

    Ui::kcaview_base ui_kcaview_base;

    QHash< quint32, AuthorityFlags* > m_authoritiesFlags;
    QHash< quint32, QTreeWidgetItem* > m_authoritiesTree;
    QHash< quint64, QTreeWidgetItem* > m_certificatesList;

    quint32 m_authority;
    quint64 m_certificate;
};

#endif // kcaVIEW_H
