/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010, 2011 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef KCADBUSTYPES_H
#define KCADBUSTYPES_H

#include <QtCore/QString>
#include <QtDBus/QDBusArgument>

template< typename T > class QList;

namespace kCA {
namespace DBus {

/** List of status codes used as parameter status for D-Bus signals. */
enum StatusCode {
  OK = 0,             /**<No error or operation successful. */
  DatabaseError,      /**<Database failed during query. */
  AuthorityNotFound,  /**<Specified authority not found. */
  AuthorityLocked,    /**<Specified authority was not unlocked while that was required. */
  SerialError,        /**<Specified serial number not found. */
  FileError,          /**<Specified filename could not be used for intended operation. */
  ParameterError,     /**<Specified parameters do not match constraints. */
  CertificateExpired, /**<Specified certificate has expired. */
  CertificateRevoked, /**<Specified certificate was revoked. */
  OtherError,         /**<Unspecified error during operation. */
};

/** Turn a StatusCode into readable text. */
inline QString parseCode(const int source) {
  switch((StatusCode) source) {
    case OK:
      return QObject::tr("No error occurred", "kCA::DBus::ErrorCode");
    case DatabaseError:
      return QObject::tr("Database backend error occurred", "kCA::DBus::ErrorCode");
    case AuthorityNotFound:
      return QObject::tr("Specified authority not found", "kCA::DBus::ErrorCode");
    case AuthorityLocked:
      return QObject::tr("Authority's private key could not be loaded", "kCA::DBus::ErrorCode");
    case SerialError:
      return QObject::tr("Specified serial number not found for selected authority", "kCA::DBus::ErrorCode");
    case FileError:
      return QObject::tr("Error during file read/write operation", "kCA::DBus::ErrorCode");
    case ParameterError:
      return QObject::tr("Given parameters do not match constraints", "kCA::DBus::ErrorCode");
    case CertificateExpired:
      return QObject::tr("Selected certificate has expired", "kCA::DBus::ErrorCode");
    case CertificateRevoked:
      return QObject::tr("Selected certificate was revoked", "kCA::DBus::ErrorCode");
    case OtherError:
      return QObject::tr("Unknown error occurred", "kCA::DBus::ErrorCode");
    default:
      return QObject::tr("Unknown code", "kCA::DBus::ErrorCode");
  }
}

/**
 * Typename for one authority entry in property authorities.
 */
struct Authority {
  quint32 hash;
  QString name;
  QString subject;
};

}; // End namespace DBus
}; // End namespace kCA

inline QDBusArgument& operator<<(QDBusArgument& argument, const kCA::DBus::Authority& source)
{
  argument.beginStructure();
  argument << source.hash << source.name << source.subject;
  argument.endStructure();

  return argument;
}

inline const QDBusArgument& operator>>(const QDBusArgument& argument, kCA::DBus::Authority& destination)
{
  argument.beginStructure();
  argument >> destination.hash >> destination.name >> destination.subject;
  argument.endStructure();

  return argument;
}

Q_DECLARE_METATYPE(kCA::DBus::Authority)
Q_DECLARE_METATYPE(QList< kCA::DBus::Authority >)

#endif // KCADBUSTYPES_H
