/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2011, 2013 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "sqlite_db_upgrade.h"

#include "sqlite_db.h"

#include <QtCore/QFile>
#include <QtCore/QTemporaryFile>

#include <string>
#include <unistd.h>

#include <sqlite3.h>

using namespace kCA::Database;

const std::string SQLiteDatabaseUpgrade::tableAuthorities[] = {
  std::string("SELECT id, hash, name, digest, certificatedays, policy, crlnumber, crldays, certificate, key "
                "FROM Authorities ORDER BY id ASC;"),
};
const std::string SQLiteDatabaseUpgrade::tablePolicies[] = {
  std::string("SELECT id, name, Country, STate, Location, Organization, OrganizationalUnit, CommonName, Email "
                "FROM Policies ORDER BY id ASC;"),
};
const std::string SQLiteDatabaseUpgrade::tableExtensions[] = {
  std::string("SELECT id, type, critical, oid, value FROM Extensions ORDER BY id ASC;"),
};
const std::string SQLiteDatabaseUpgrade::tableCAExtensions[] = {
  std::string("SELECT id, authority, extension FROM CAExtensions ORDER BY id ASC;"),
};
const std::string SQLiteDatabaseUpgrade::tableCertificates[] = {
  std::string("SELECT id, serial, fingerprint, authority, subject, issued, validfrom, expiration, "
                  "owner, status, revocation, reason, certificate "
                "FROM Certificates ORDER BY id ASC;"),
};

SQLiteDatabaseUpgrade::SQLiteDatabaseUpgrade(const QString& filename, const char* targetVersion, QObject* parent) :
    QThread(parent),
    upgradeNeeded(false),
    upgradeReady(false),
    oldversion(0),
    dbFile(filename),
    source(0),
    destination(0)
{
  if (sqlite3_open_v2(filename.toUtf8().constData(), &source,
                      SQLITE_OPEN_READONLY | SQLITE_OPEN_FULLMUTEX | SQLITE_OPEN_PRIVATECACHE,
                      NULL) != SQLITE_OK || !source) {
    // Abort operation
    return;
  }

  sqlite3_stmt *fetchVersion;
  sqlite3_prepare_v2(source, "SELECT value FROM Settings WHERE name = 'version';", 50, &fetchVersion, NULL);
  if (sqlite3_step(fetchVersion) == SQLITE_ROW) {
    int bufsize = sqlite3_column_bytes(fetchVersion, 0);
    const char* buf =(const char*) sqlite3_column_text(fetchVersion, 0);

    // If targetVersion is same as current database version, abort.
    if (strncmp(buf, targetVersion, bufsize) == 0) {
      sqlite3_finalize(fetchVersion);
      return;
    }

    oldversion = QString(buf).toInt();
  }
  sqlite3_finalize(fetchVersion);

  upgradeNeeded = true;
}

SQLiteDatabaseUpgrade::~SQLiteDatabaseUpgrade()
{
  if (source)
    sqlite3_close(source);

  if (destination)
    sqlite3_close(destination);
}

bool SQLiteDatabaseUpgrade::upgradeRequired() const
{
  return upgradeNeeded;
}

bool SQLiteDatabaseUpgrade::prepareUpgrade()
{
  QString upgradedDb(dbFile);
  upgradedDb.append(".new");

  // Create new database
  SQLiteDatabase::init(upgradedDb);

  if (sqlite3_open_v2(upgradedDb.toUtf8().constData(), &destination,
                      SQLITE_OPEN_READWRITE | SQLITE_OPEN_FULLMUTEX | SQLITE_OPEN_PRIVATECACHE,
                      NULL) != SQLITE_OK || !destination) {
    // Abort operation
    unlink(upgradedDb.toUtf8().constData());
    return false;
  }

  upgradeReady = ((oldversion > 0) && (source) && (destination));
  return upgradeReady;
}

void SQLiteDatabaseUpgrade::run()
{
  // If upgrade not needed or not prepared, do not continue
  if (!upgradeReady || !upgradeNeeded)
    exit(1);

  // Array index begins at 0, decrease version.
  --oldversion;

  std::string statement;
  sqlite3_stmt *fetch, *store;

  // Table Authorities
  statement = tableAuthorities[oldversion];
  sqlite3_prepare_v2(source, statement.c_str(), statement.length(), &fetch, NULL);

  statement = std::string("INSERT INTO Authorities (id, hash, name, digest, certificatedays,"
                                "policy, crlnumber, crldays, certificate, key)"
                              "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?);");
  sqlite3_prepare_v2(destination, statement.c_str(), statement.length(), &store, NULL);
  while (sqlite3_step(fetch) == SQLITE_ROW) {
    sqlite3_bind_int(store, 1, sqlite3_column_int(fetch, 0));
    sqlite3_bind_int64(store, 2, sqlite3_column_int64(fetch, 1));
    sqlite3_bind_text(store, 3,(const char*) sqlite3_column_text(fetch, 2), sqlite3_column_bytes(fetch, 2), SQLITE_STATIC);
    sqlite3_bind_int(store, 4, sqlite3_column_int(fetch, 3));
    sqlite3_bind_int(store, 5, sqlite3_column_int(fetch, 4));
    sqlite3_bind_int(store, 6, sqlite3_column_int(fetch, 5));
    sqlite3_bind_int(store, 7, sqlite3_column_int(fetch, 6));
    sqlite3_bind_int(store, 8, sqlite3_column_int(fetch, 7));
    sqlite3_bind_blob(store, 9, sqlite3_column_blob(fetch, 8), sqlite3_column_bytes(fetch, 8), SQLITE_STATIC);
    sqlite3_bind_blob(store, 10, sqlite3_column_blob(fetch, 9), sqlite3_column_bytes(fetch, 9), SQLITE_STATIC);

    sqlite3_step(store);
    sqlite3_reset(store);
  }
  sqlite3_finalize(store);
  sqlite3_finalize(fetch);

  // Table Policies
  statement = tablePolicies[oldversion];
  sqlite3_prepare_v2(source, statement.c_str(), statement.length(), &fetch, NULL);

  statement = std::string("INSERT INTO Policies (id, name, Country, STate, Location, Organization,"
                                "OrganizationalUnit, CommonName, Email)"
                              "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?);");
  sqlite3_prepare_v2(destination, statement.c_str(), statement.length(), &store, NULL);
  while (sqlite3_step(fetch) == SQLITE_ROW) {
    sqlite3_bind_int(store, 1, sqlite3_column_int(fetch, 0));
    sqlite3_bind_text(store, 2,(const char*) sqlite3_column_text(fetch, 1), sqlite3_column_bytes(fetch, 1), SQLITE_STATIC);
    sqlite3_bind_int(store, 3, sqlite3_column_int(fetch, 2));
    sqlite3_bind_int(store, 4, sqlite3_column_int(fetch, 3));
    sqlite3_bind_int(store, 5, sqlite3_column_int(fetch, 4));
    sqlite3_bind_int(store, 6, sqlite3_column_int(fetch, 5));
    sqlite3_bind_int(store, 7, sqlite3_column_int(fetch, 6));
    sqlite3_bind_int(store, 8, sqlite3_column_int(fetch, 7));
    sqlite3_bind_int(store, 9, sqlite3_column_int(fetch, 8));

    sqlite3_step(store);
    sqlite3_reset(store);
  }
  sqlite3_finalize(store);
  sqlite3_finalize(fetch);

  // Table Extensions
  statement = tableExtensions[oldversion];
  sqlite3_prepare_v2(source, statement.c_str(), statement.length(), &fetch, NULL);

  statement = std::string("INSERT INTO Extensions (id, type, critical, oid, value)"
                              "VALUES (?, ?, ?, ?, ?);");
  sqlite3_prepare_v2(destination, statement.c_str(), statement.length(), &store, NULL);
  while (sqlite3_step(fetch) == SQLITE_ROW) {
    sqlite3_bind_int(store, 1, sqlite3_column_int(fetch, 0));
    sqlite3_bind_int(store, 2, sqlite3_column_int(fetch, 1));
    sqlite3_bind_int(store, 3, sqlite3_column_int(fetch, 2));
    sqlite3_bind_text(store, 4,(const char*) sqlite3_column_text(fetch, 3), sqlite3_column_bytes(fetch, 3), SQLITE_STATIC);
    sqlite3_bind_text(store, 5,(const char*) sqlite3_column_text(fetch, 4), sqlite3_column_bytes(fetch, 4), SQLITE_STATIC);

    sqlite3_step(store);
    sqlite3_reset(store);
  }
  sqlite3_finalize(store);
  sqlite3_finalize(fetch);

  // Table CAExtensions
  statement = tableCAExtensions[oldversion];
  sqlite3_prepare_v2(source, statement.c_str(), statement.length(), &fetch, NULL);

  statement = std::string("INSERT INTO CAExtensions (id, authority, extension)"
                              "VALUES (?, ?, ?);");
  sqlite3_prepare_v2(destination, statement.c_str(), statement.length(), &store, NULL);
  while (sqlite3_step(fetch) == SQLITE_ROW) {
    sqlite3_bind_int(store, 1, sqlite3_column_int(fetch, 0));
    sqlite3_bind_int(store, 2, sqlite3_column_int(fetch, 1));
    sqlite3_bind_int(store, 3, sqlite3_column_int(fetch, 2));

    sqlite3_step(store);
    sqlite3_reset(store);
  }
  sqlite3_finalize(store);
  sqlite3_finalize(fetch);

  // Table Certificates
  statement = tableCertificates[oldversion];
  sqlite3_prepare_v2(source, statement.c_str(), statement.length(), &fetch, NULL);

  statement = std::string("INSERT INTO Certificates (id, serial, fingerprint, authority, subject,"
                                "issued, validfrom, expiration, owner, status, revocation,"
                                "reason, certificate)"
                              "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);");
  sqlite3_prepare_v2(destination, statement.c_str(), statement.length(), &store, NULL);
  while (sqlite3_step(fetch) == SQLITE_ROW) {
    sqlite3_bind_int(store, 1, sqlite3_column_int(fetch, 0));
    sqlite3_bind_int64(store, 2, sqlite3_column_int64(fetch, 1));
    sqlite3_bind_text(store, 3,(const char*) sqlite3_column_text(fetch, 2), sqlite3_column_bytes(fetch, 2), SQLITE_STATIC);
    sqlite3_bind_int64(store, 4, sqlite3_column_int64(fetch, 3));
    sqlite3_bind_text(store, 5,(const char*) sqlite3_column_text(fetch, 4), sqlite3_column_bytes(fetch, 4), SQLITE_STATIC);
    sqlite3_bind_text(store, 6,(const char*) sqlite3_column_text(fetch, 5), sqlite3_column_bytes(fetch, 5), SQLITE_STATIC);
    sqlite3_bind_text(store, 7,(const char*) sqlite3_column_text(fetch, 6), sqlite3_column_bytes(fetch, 6), SQLITE_STATIC);
    sqlite3_bind_text(store, 8,(const char*) sqlite3_column_text(fetch, 7), sqlite3_column_bytes(fetch, 7), SQLITE_STATIC);
    sqlite3_bind_text(store, 9,(const char*) sqlite3_column_text(fetch, 8), sqlite3_column_bytes(fetch, 8), SQLITE_STATIC);
    sqlite3_bind_int(store, 10, sqlite3_column_int(fetch, 9));
    sqlite3_bind_text(store, 11,(const char*) sqlite3_column_text(fetch, 10), sqlite3_column_bytes(fetch, 10), SQLITE_STATIC);
    sqlite3_bind_int(store, 12, sqlite3_column_int(fetch, 11));
    sqlite3_bind_blob(store, 13, sqlite3_column_blob(fetch, 12), sqlite3_column_bytes(fetch, 12), SQLITE_STATIC);

    sqlite3_step(store);
    sqlite3_reset(store);
  }
  sqlite3_finalize(store);
  sqlite3_finalize(fetch);

  // Process completed, move files.
  sqlite3_close(destination);
  sqlite3_close(source);

  destination = 0;
  source = 0;

  unlink(dbFile.toUtf8().constData());
  QFile::rename(dbFile + ".new", dbFile);

  exit(0);
}
