/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef AUTHORITY_H
#define AUTHORITY_H

#include <libkca_ossl/commons.h>

#include <QtCore/QList>
#include <QtCore/QObject>

template<typename T> class QList;
template<typename Key, typename T> class QHash;

class QByteArray;
class QDateTime;
class QString;
class QStringList;
class QUrl;

namespace kCA_ossl
{
class Certificate;
class Request;
class Revocationlist;
class X509Extension;
class X509Name;
};

namespace kCA
{
namespace Database
{
struct Certificate;
struct Policy;

class Database;
}

class AuthorityPrivate;


/**
 * Manages a certification authority and takes care of necessary
 * database operations.
 * It signs certification requests, issues certificate revocation
 * lists and takes care of revoking certificates.
 *
 * @short Certification Authority management class.
 * @author Felix Tiede <info@pc-tiede.de>
 */
class Authority : public QObject
{
  friend class AuthorityPrivate;

  Q_OBJECT
  public:
    /** Error codes for authority operations. */
    enum AuthorityError {
      NoError,          /**<Operation successful. */
      KeyNotLoaded,     /**<Private key was not loaded or unlocked. */
      DatabaseError,    /**<Problem accessing database. */
      ConstraintError,  /**<Constraints of parameters were not matched. */
      NotFound,         /**<Requested data was not found in database. */
      TimeoutError,     /**<Operation timed out. */
      RandomizerError,  /**<Error in RNG during serial number generation. */
      SigningError,     /**<Error in signing by libkca_ossl. */
    };

    /**
     * Create instance from given database and hash value.
     * @param database Instance of database backend
     * @param hash Hash value of authority to load from database
     * @param parent Parent of instance
     */
    Authority(Database::Database* database, const quint32 hash, QObject* parent = 0);
    Authority(Database::Database* database, const QString& name, const Database::Policy& policy,
              const kCA_ossl::Certificate& certificate,
              const QByteArray& key, QObject* parent = 0);
    ~Authority();

    /**
     * Get usability for signing processes.
     * Returns false if authority certificate is not yet valid, expired or revoked.
     */
    bool enabled() const;

    /** Get subject hash value of certificate, the instance uses for signing. */
    quint32 hash() const;
    /** Get issuer hash value of certificate, the instance uses for signing. Identical with hash() if root CA. */
    quint32 issuerHash() const;

    /** Get friendly name of certification authority. */
    QString name() const;
    /** Set friendly name of certification authority. */
    void setName(const QString& value);
    /** Get authority's subject. */
    QString subject() const;

    /** Get default validity time of certificates in days. */
    unsigned int certificateDays() const;
    /** Set default validity time of certificates in days. */
    void setCertificateDays(unsigned int value);
    /** Get default validity time of certificate revocation lists in days. */
    unsigned int crlDays() const;
    /** Set default validity time of certificate revocation lists in days. */
    void setCrlDays(unsigned int value);

    /** Get current digest setting used for all signing operations. */
    kCA_ossl::Digest digest() const;
    /** Set digest for all forthcoming signing operations. */
    void setDigest(kCA_ossl::Digest digest);

    /** Get CRL distribution point for authority. */
    QUrl crlDistributionPoint() const;
    /** Set CRL distribution point for authority, write protected, if crlDistributionPoint() is non-empty. */
    void setCrlDistributionPoint(const QUrl& value);

    /** Get match policy for authority. */
    Database::Policy getPolicy() const;
    /** Set match policy for authority. */
    void setPolicy(const Database::Policy& policy);

    /** Get list of extensions used for certificates issued. */
    QList<const kCA_ossl::X509Extension*> certificateExtensions() const;
    /** Set list of extensions for certificates issued. */
    void setCertificateExtensions(const QList<const kCA_ossl::X509Extension*>& extensions);

    /** Get ist of extensions used for revocation lists issued. */
    QList<const kCA_ossl::X509Extension*> crlExtensions() const;
    /** Set list of extensions for revocation lists issued. */
    void setCrlExtensions(const QList<const kCA_ossl::X509Extension*>& extensions);

    /** Get certificates associated with authority. */
    const QHash< quint64, Database::Certificate > certificates() const;

    /** Verify if X.509 name matches authority's policy. */
    bool matchPolicy(const kCA_ossl::X509Name& name) const;

    /** If database contains unencrypted key, this returns false. */
    bool lockableKey() const;
    /**
     * Load and unlock private key.
     * @param password Password needed to unlock encrypted private key.
     * @param keep If true, private key is not unloaded automatically after next key-bound operation.
     * @param location Set new location of private key, only stored if key was successfully loaded.
     * @param store If true, new location will be stored in database.
     * @return True, if private key was successfully unlocked, otherwise false.
     */
    bool unlockKey(const QString& password = QString(), const bool keep = false,
                   const QString& location = QString(), const bool store = false);
    /** Unload and lock private key, reverts unlockKey. */
    void lockKey();
    /** Get lock/unlock state of private key. */
    bool keyUnlocked() const;
    /** Get stored private key filename unless (encrypted) key is stored in database. */
    QString keyPath() const;

    /**
     * Replace authority certificate and private key.
     * If key is empty, passphrase is used to encrypt private key from certificate
     * Empty passphrase will store an unencrypted private key in the database. If key is not
     * empty it must contain either PEM-encoded key data or a filename to a private key file
     * to which passphrase is the decoding passphrase.
     * If key does not work as private key for certificate, the method will fail.
     * keyUnlocked() must return true for this method to work.
     * @param certificate New certificate to use for authority.
     * @param key Pathname or PEM-encoded raw key data, defaults to empty.
     * @return True if certificate replacement was successful, otherwise false.
     */
    bool replaceCertificate(const kCA_ossl::Certificate& certificate,
                            const QByteArray& key = QByteArray(),
                            const QString& passphrase = QString());

    /** Get last error message from database backend. */
    QString databaseError() const;

    /** Find all authorities in configfile and tell their names. */
    static QStringList findAuthorities(const QString& configfile);
    /** Import name certification authority from configfile to database. */
    static bool importAuthority(Database::Database* database, const QString& configfile, const QString& name,
                                const bool importKey);

  public slots:
    /**
     * Sign request and generate Certificate with given validity timestamps.
     * Generated certificate is returned via signal requestSigned().
     * @param owner Name of owner of certificate.
     * @param request Certificate signing request to be signed.
     * @param starttime Begin of validity timespan, defaults to now if empty.
     * @param endtime End of validity timespan, defaults to now certificateDays() if empty.
     * @param extensions List of extensions which shall be set along with certificateExtensions().
     * @param supersede If true all currently valid certificates with same subject hash are superseded, defaults to true.
     * @param timeout Stop certificate generation after timeout seconds, defaults to 5.
     */
    void signRequest(const QString& owner, const kCA_ossl::Request& request,
                     const QDateTime& starttime, const QDateTime& endtime,
                     const QList< const kCA_ossl::X509Extension* >& extensions = QList< const kCA_ossl::X509Extension* >(),
                     const bool supersede = true, const unsigned int timeout = 5);
    /**
     * Revoke certificate with given reason and timestamp.
     * @param serial Serial number of certificate to revoke.
     * @param reason Reason of certificate revocation.
     * @param timestamp Timestamp of certificate revocation, default to now if empty.
     * @return Treu, if revocation was successful, false, otherwise.
     */
    void revokeCertificate(const quint64 serial, const kCA_ossl::RevocationReason reason,
                           const QDateTime& timestamp);

    /**
     * Generate certificate revocation list for authority and store as filename, using extensions if given.
     * Generated certificate revocation list is returned via signal crlGenerated().
     * @param days Validity timespan of next CRL, defaults to crlDays().
     * @param extensions X509 extensions to use for the next CRL, defaults to crlExtensions().
     */
    void generateCrl(const unsigned int days = 0,
                     const QList<const kCA_ossl::X509Extension*>& extensions = QList<const kCA_ossl::X509Extension*>());

  signals:
    /** Emitted when authority's key is locked.
     * @param hash Subject hash value of authority sending the signal.
     */
    void keyLocked(const quint32 hash);
    /** Emitted when authority's key is unlocked.
     * @param hash Subject hash value of authority sending the signal.
     */
    void keyUnlocked(const quint32 hash);

    /** Emitted when a request is signed or signing failed permanently. */
    void requestSigned(const kCA_ossl::Certificate& certificate, const int error);
    /** Emitted when a certificate was revoked. */
    void certificateRevoked(const quint32 hash, const quint64 serial, const int reason, const int error);

    /** Emitted when a certificate revocation list has been completed. */
    void crlGenerated(const kCA_ossl::Revocationlist& crl, const int error);

  private:
    void setCrlNumber(ulong value);

    AuthorityPrivate * const d;
};

};

#endif // AUTHORITY_H
