/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _LIBKCAOSSL_X509NAME_H
#define _LIBKCAOSSL_X509NAME_H

#include "kca_ossl_export.h"

#include <openssl/x509.h>

#include <QtCore/QObject>
#include <QtCore/QMultiMap>

struct X509_name_st;
typedef struct X509_name_st X509_NAME;

class QString;
class QStringList;

/** Namespace for kca-local classes */
namespace kCA_ossl
{

class Certificate;
class Request;
class RequestPrivate;
class Revocationlist;
class RevocationlistPrivate;

/**
 * Represents a X509 name.
 * Basically adds an abstraction layer over OpenSSL X509_NAME.
 *
 * @short OpenSSL X509_NAME abstraction class.
 * @author Felix Tiede <info@pc-tiede.de>
 */
class _LIBKCAOSSL_EXPORT X509Name : public QObject
{
  friend class Certificate;
  friend class Request;
  friend class RequestPrivate;
  friend class Revocationlist;
  friend class RevocationlistPrivate;

  Q_OBJECT
  public:
    /** Creates an empty X509 name. */
    X509Name(QObject *parent = 0);
    /** Copy constructor */
    X509Name(const X509Name &other, QObject *parent = 0);
    /**
     * Creates a X509 name.
     * Values are taken of QMultiMap\<QString, QString\>, where the
     * key must be a known OID. Values for unknown OIDs can be added
     * later using addValues(const QString&, const QString&, const QString&, const QStringList&).
     * OIDs can also be added using Utils::registerOID().
     *
     * @param values Values to store for this name.
     * @param parent Owner object of this instance, default 0.
     */
    X509Name(const QMultiMap< QString, QString > &values, QObject *parent = 0);
    /**
     * Creates a X509 name.
     * Values are taken from string parameter, which must be of form
     * @code
     * /?<OID>=<value>(/<OID>=<value>)*
     * @endcode
     *
     * @param name String to parse into a X509Name.
     * @param parent Owner object of this instance, default 0.
     */
    X509Name(const QString &name, QObject *parent = 0);
    ~X509Name();

    /**
     * Get stored keys, which are short names of used OIDs.
     */
    const QStringList keys() const;
    /**
     * Get stored values for given key, which is the short name of an OID.
     * Available keys can be retrieved with keys().
     *
     * @param key Short name of OID for which values should be retrieved.
     * @return List of QStrings with values if successful, empty list otherwise.
     */
    const QStringList values(const QString &key) const;

    /**
     * Sets data with object identifier. String must be either a numerical OID,
     * a short or long name of an object identifier known to OpenSSL.
     *
     * @param oid Numerical, short or long name of a known OID.
     * @param values Values to be set for the given OID.
     */
    void addValues(const QString &oid, const QStringList &values);
    /**
     * Sets data with an object identifier unknown to OpenSSL.
     *
     * @param oid Numerical representation of OID.
     * @param longName Long name of OID.
     * @param shortName Short name of OID.
     * @param values Values to be set for the given OID.
     */
    void addValues(const QString &oid, const QString &shortName, const QString &longName,
                   const QStringList &values);

    /**
     * Converts the name into one concatenated string.
     */
    const QString toString() const;

    quint32 hash() const;

    /** Assigns other to this instance. */
    X509Name &operator=(const X509Name &other);

  protected:
    /**
     * Create extension from OpenSSL's internal extension representation,
     * X509_NAME.
     *
     * @param other X509_NAME struct to be parsed.
     * @param parent Owner object of this instance, default 0.
     */
    X509Name(X509_NAME *other, QObject *parent = 0);

    /**
     * Gets OpenSSL's internal representation of name.
     */
    X509_NAME *internal() const;

  private:
    QMultiMap< QString, QString > data;

    X509_NAME *osslData;

    /**
     * Internal function to fill data from *internal.
     */
    void parseToMap();
};

};

#endif // _LIBKCAOSSL_X509NAME_H
