/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "x509name.h"

#include "utils.h"

#include <openssl/asn1.h>
#include <openssl/objects.h>
#include <openssl/ossl_typ.h>
#include <openssl/x509.h>

#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QMultiMap>

using namespace kCA_ossl;

X509Name::X509Name(QObject *parent) :
    QObject(parent),
    osslData(0)
{}

X509Name::X509Name(const X509Name &other, QObject *parent) :
    QObject(parent),
    data(other.data),
    osslData(0)
{
  if (other.osslData) {
    osslData = X509_NAME_dup(other.osslData);
  }
}

X509Name::X509Name(X509_NAME* other, QObject* parent) :
    QObject(parent),
    osslData(0)
{
  if (other) {
    osslData = X509_NAME_dup(other);
    parseToMap();
  }
}

X509Name::X509Name(const QMultiMap< QString, QString > &values, QObject *parent) : QObject(parent)
{
  osslData = X509_NAME_new();
  if (!osslData) {
    return;
  }

  bool incomplete = false;
  int nid;
  foreach(QString key, values.keys()) {
    nid = Utils::nid(key);
    if (nid == NID_undef) {
      incomplete = true;
      continue;
    }
    foreach(QString value, values.values(key)) {
      if (!X509_NAME_add_entry_by_NID(osslData, nid, MBSTRING_UTF8,
              (unsigned char *) value.toUtf8().constData(), value.length(), -1, 0)) {
        incomplete = true;
      }
    }
  }

  // Parsing is expensive. only do it if needed
  if (incomplete) {
    parseToMap();
  }
  else {
    data = values;
  }
}

X509Name::X509Name(const QString &name, QObject *parent) :
    QObject(parent),
    osslData(X509_NAME_new())
{
  if (!osslData) {
    return;
  }

  QString key;
  QStringList ne, values;

  foreach(QString entry, name.split("/", QString::SkipEmptyParts)) {
    ne = entry.split("=", QString::SkipEmptyParts);
    if (ne.size() < 2) {
      continue;
    }
    else {
      key = ne[0];
      values = ne.mid(1);
    }

    foreach(QString value, values) {
      if (X509_NAME_add_entry_by_NID(osslData, Utils::nid(key), MBSTRING_UTF8,
            (unsigned char *) value.toUtf8().constData(), value.length(), -1, 0)) {
        data.insertMulti(key, value);
      }
    }
  }
}

X509Name::~X509Name()
{
  data.clear();

  if (osslData) {
    X509_NAME_free(osslData);
  }
}

const QStringList X509Name::keys() const
{
  return data.uniqueKeys();
}

const QStringList X509Name::values(const QString& key) const
{
  if (data.contains(key)) {
    return data.values(key);
  }

  return QStringList();
}

X509_NAME *X509Name::internal() const
{
  if (osslData) {
    return X509_NAME_dup(osslData);
  }

  return 0;
}

void X509Name::addValues(const QString &oid, const QStringList &values)
{
  if (!osslData) {
    osslData = X509_NAME_new();
  }

  int nid = Utils::nid(oid);
  if (nid == NID_undef) {
    return;
  }

  foreach(QString value, values) {
    if (X509_NAME_add_entry_by_NID(osslData, nid, MBSTRING_UTF8,
          (unsigned char *) value.toUtf8().constData(), value.length(), -1, 0)) {
      data.insertMulti(OBJ_nid2sn(nid), value);
    }
  }
}

void X509Name::addValues(const QString &oid, const QString &shortName,
                         const QString &longName, const QStringList &values)
{
  if (!osslData) {
    osslData = X509_NAME_new();
  }

  int nid;
  if (oid.length() == 0) {
    nid = Utils::nid(oid);
  }
  else {
    nid = Utils::nid(shortName);
  }

  if (nid == NID_undef) {
    nid = Utils::registerOid(oid.toLocal8Bit().constData(),
                     shortName.toLocal8Bit().constData(),
                     longName.toLocal8Bit().constData());
  }

  ASN1_OBJECT *obj = OBJ_nid2obj(nid);
  if (!obj) {
    return;
  }

  foreach(QString value, values) {
    if (X509_NAME_add_entry_by_NID(osslData, nid, MBSTRING_UTF8,
          (unsigned char *) value.toUtf8().constData(), value.length(), -1, 0)) {
      data.insertMulti(shortName, value);
    }
  }
}

quint32 X509Name::hash() const
{
  if (osslData)
    return X509_NAME_hash(osslData);

  return 0;
}

const QString X509Name::toString() const
{
  static const QStringList keyNames(QStringList() << "C" << "ST" << "L" << "O" << "OU" << "CN");

  QStringList keys = keyNames;
  foreach(QString key, data.uniqueKeys()) {
    if (!keyNames.contains(key)) {
      keys.append(key);
    }
  }

  QString result = "/";
  foreach(QString key, keys) {
    foreach(QString value, data.values(key)) {
      if (!value.isEmpty()) {
        result += key + "=" + value + "/";
      }
    }
  }


  return result;
}

X509Name &X509Name::operator=(const X509Name &other)
{
  data = other.data;
  if (other.osslData) {
    osslData = X509_NAME_dup(other.osslData);
  }
  else {
    osslData = X509_NAME_new();
  }

  return *this;
}

void X509Name::parseToMap()
{
  if (!osslData) {
    return;
  }

  X509_NAME_ENTRY* ne;
  ASN1_OBJECT* obj;
  ASN1_STRING* value;

  for (int i=0; i<X509_NAME_entry_count(osslData); ++i)
  {
    ne  = X509_NAME_get_entry(osslData, i);
    obj = X509_NAME_ENTRY_get_object(ne);
    value = X509_NAME_ENTRY_get_data(ne);

    data.insertMulti(OBJ_nid2sn(OBJ_obj2nid(obj)),
                      (const char *) ASN1_STRING_data(value));
  }
}
