/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "x509extension.h"

#include "utils.h"

#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/buffer.h>
#include <openssl/objects.h>
#include <openssl/ossl_typ.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#include <QtCore/QByteArray>
#include <QtCore/QString>

using namespace kCA_ossl;

X509Extension::X509Extension(QObject *parent) :
    QObject(parent),
    nid(NID_undef),
    critical(false),
    valid(false),
    osslData(0)
{}

X509Extension::X509Extension(const X509Extension &other, QObject *parent) :
    QObject(parent),
    nid(other.nid),
    critical(other.critical),
    valid(other.valid),
    data(other.data),
    osslData(0)
{
  if (other.osslData)
    osslData = X509_EXTENSION_dup(other.osslData);
}

X509Extension::X509Extension(X509_EXTENSION *other, QObject *parent) :
    QObject(parent),
    nid(NID_undef),
    critical(false),
    valid(false),
    osslData(0)
{
  if (other && other->object && other->value) {
    osslData = X509_EXTENSION_dup(other);
    nid = OBJ_obj2nid(osslData->object);
    critical = (X509_EXTENSION_get_critical(osslData) != 0);

    BIO *memory = BIO_new(BIO_s_mem());
    if (memory && X509V3_EXT_print(memory, osslData, 0, 0))
    {
      BUF_MEM *bptr;
      BIO_get_mem_ptr(memory, &bptr);
      data = QByteArray(bptr->data, bptr->length);
      BIO_free(memory);
      valid = true;
    }
  }
}

X509Extension::X509Extension(const QString &oid, const QByteArray &value, bool critical,
                             QObject *parent) :
    QObject(parent),
    nid(NID_undef),
    critical(critical),
    valid(false),
    data(value),
    osslData(0)
{
  nid = Utils::nid(oid);
  if (nid == NID_undef) {
    return;
  }

  // Try to create X509_EXTENSION pointer, it may fail for some X.509V3 extensions,
  // which is no problem by itself.
  osslData = X509V3_EXT_conf_nid(NULL, NULL, nid,(char *) value.constData());
  if (osslData) {
    X509_EXTENSION_set_critical(osslData, (critical ? 1 : 0));
  }
  else {
    // If there's no valid X509_EXTENSION pointer, criticality must be stored in the value.
    if (critical && !value.contains("critical")) {
      data = QByteArray("critical, ").append(value);
    }
  }

  valid = true;
}

X509Extension::X509Extension(const QString &oid, const QString &oidShortName,
                             const QString &oidLongName, const QByteArray &value,
                             bool critical, QObject *parent) :
    QObject(parent),
    nid(NID_undef),
    critical(critical),
    valid(false),
    data(value),
    osslData(0)
{
  if (oid.length() == 0) {
    nid = Utils::nid(oidShortName);
  }
  else {
    nid = Utils::nid(oid);
  }

  if (nid == NID_undef) {
    nid = Utils::registerOid(oid.toLocal8Bit().constData(),
                             oidShortName.toLocal8Bit().constData(),
                             oidLongName.toLocal8Bit().constData());
  }

  // Try to create X509_EXTENSION pointer, it may fail for some X.509V3 extensions,
  // which is no problem by itself.
  osslData = X509V3_EXT_conf_nid(NULL, NULL, nid,(char *) value.constData());
  if (osslData) {
    X509_EXTENSION_set_critical(osslData, (critical ? 1 : 0));
  }
  else {
    // If there's no valid X509_EXTENSION pointer, criticality must be stored in the value.
    if (critical && !value.contains("critical")) {
      data = QByteArray("critical, ").append(value);
    }
  }

  valid = true;
}

X509Extension::~X509Extension()
{
  valid = false;
  if (osslData)
    X509_EXTENSION_free(osslData);
}

const QString X509Extension::oid() const
{
  return Utils::oid(nid);
}

const QString X509Extension::oidShortName() const
{
  const char *buf = OBJ_nid2sn(nid);
  if (buf) {
    return QString(buf);
  }

  return QString();
}

const QString X509Extension::oidLongName() const
{
  const char *buf = OBJ_nid2ln(nid);
  if (buf) {
    return QString(buf);
  }

  return QString();
}

const QByteArray &X509Extension::value() const
{
  return data;
}

bool X509Extension::isValid() const
{
  return valid;
}

bool X509Extension::isCritical() const
{
  return critical;
}

int X509Extension::getNid() const
{
  return nid;
}

X509_EXTENSION *X509Extension::internal() const
{
  if (valid) {
    return X509_EXTENSION_dup(osslData);
  }

  return 0;
}

void X509Extension::setOid(const QString &oid)
{
  if (!osslData)
    osslData = X509_EXTENSION_new();

  int nid = Utils::nid(oid);
  if (nid == NID_undef) {
    return;
  }

  ASN1_OBJECT *obj = OBJ_nid2obj(nid);
  if (!obj) {
    return;
  }

  this->nid = nid;
  X509_EXTENSION_set_object(osslData, obj);
  ASN1_OBJECT_free(obj);
  valid = true;
}

void X509Extension::setOid(const QString &oid, const QString &shortName, const QString &longName)
{
  if (!osslData)
    osslData = X509_EXTENSION_new();

  int nid = Utils::nid(oid);
  if (nid == NID_undef) {
    nid = OBJ_create(oid.toLocal8Bit().constData(),
                     shortName.toLocal8Bit().constData(),
                     longName.toLocal8Bit().constData());
  }
  else {
    int snNID = Utils::nid(shortName);
    int lnNID = Utils::nid(longName);
    if (!(snNID != NID_undef && snNID == lnNID && snNID == nid)) {
      return;
    }
  }

  ASN1_OBJECT *obj = OBJ_nid2obj(nid);
  if (!obj) {
    return;
  }

  this->nid = nid;
  X509_EXTENSION_set_object(osslData, obj);
  ASN1_OBJECT_free(obj);
  valid = true;
}

void X509Extension::setValue(const QByteArray &value)
{
  if (!osslData)
    osslData = X509_EXTENSION_new();

  ASN1_OCTET_STRING *data = ASN1_OCTET_STRING_new();
  if (!data) {
    return;
  }

  this->data = value;

  ASN1_OCTET_STRING_set(data,(const unsigned char *) value.constData(), value.length());
  X509_EXTENSION_set_data(osslData, data);
  ASN1_STRING_free(data);
  valid = true;
}

void X509Extension::setCritical(bool critical)
{
  if (!osslData)
    osslData = X509_EXTENSION_new();

  this->critical = critical;
  X509_EXTENSION_set_critical(osslData, (critical ? 1 : 0));
}

X509Extension &X509Extension::operator=(const X509Extension &other)
{
  critical = other.critical;
  nid = other.nid;
  data = QByteArray(other.data);
  osslData = 0;

  if (other.valid) {
    osslData = X509_EXTENSION_dup(other.osslData);
  }

  if (osslData) {
    valid = true;
  }

  return *this;
}
