/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2011 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "utils.h"

#include "commons.h"
#include "x509extension.h"

#include <QtCore/QByteArray>
#include <QtCore/QDateTime>
#include <QtCore/QList>
#include <QtCore/QLocale>
#include <QtCore/QString>

#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/bn.h>
#include <openssl/evp.h>
#include <openssl/objects.h>
#include <openssl/rand.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

using namespace kCA_ossl;

// Ciphers are not loaded yet
bool Utils::OSSL_ciphers_loaded = false;

// Filename of PRNG seed file
// TODO: correct flag between unix / windows? Can't use compiler...
#ifdef __unix__
const char *Utils::random_source = "/dev/urandom";
#else
const char *Utils::random_source = ".rnd";
#endif

// Feed "global" extension lists for standard certificates
const QList< const X509Extension* > Utils::emailCertConstraints(
    QList< const X509Extension* >() <<
    new X509Extension("subjectKeyIdentifier", "hash", false) <<
    new X509Extension("authorityKeyIdentifier", "keyid,issuer:always", false) <<
    new X509Extension("basicConstraints", "CA:false", false) <<
    new X509Extension("keyUsage", "dataEncipherment, digitalSignature, keyEncipherment",
                                false) <<
    new X509Extension("extendedKeyUsage", "emailProtection") <<
    new X509Extension("nsCertType", "email")
  );
const QList< const X509Extension* > Utils::CACertConstraints(
    QList< const X509Extension* >() <<
    new X509Extension("subjectKeyIdentifier", "hash", false) <<
    new X509Extension("authorityKeyIdentifier", "keyid:always,issuer:always", false) <<
    new X509Extension("basicConstraints", "CA:true", true) <<
    new X509Extension("keyUsage", "cRLSign, keyCertSign", false) <<
    new X509Extension("nsCertType", "sslCA, emailCA, objCA")
  );

quint64 kCA_ossl::generateRandom()
{
  union {
    quint64 number;
    unsigned char buffer[8];
  };

  RAND_load_file(Utils::random_source, 64);
  if (RAND_pseudo_bytes(buffer, 8) < 0)
    number =(quint64) ~0;

  RAND_write_file(Utils::random_source);

  return number;
}

const QList< const X509Extension* > kCA_ossl::emailCertExtensions()
{
  return Utils::emailCertConstraints;
}

QByteArray Utils::print(const ASN1_BIT_STRING* in)
{
  if (!in) {
    return QByteArray();
  }

  QByteArray result;
  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *ptr;

  if (memory && (BIO_write(memory,(char*) in->data, in->length) == in->length)) {
    BIO_get_mem_ptr(memory, &ptr);
    result = QByteArray(ptr->data, ptr->length);

    BIO_free(memory);
  }
  else {
    result = QByteArray();
  }
  return result;
}

QDateTime Utils::convert(const ASN1_TIME *in)
{
  if (!in) {
    return QDateTime();
  }

  QDateTime result;
  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *ptr;

  if (memory && ASN1_TIME_print(memory,(ASN1_TIME *) in)) {
    BIO_get_mem_ptr(memory, &ptr);
    result = QLocale::c().toDateTime(QByteArray(ptr->data, ptr->length),
                                     "MMM dd HH:mm:ss yyyy 'GMT'");
    result.setTimeSpec(Qt::UTC);
    if (!result.isValid()) {
      result = QLocale::c().toDateTime(QByteArray(ptr->data, ptr->length),
                                       "MMM  d HH:mm:ss yyyy 'GMT'");
      result.setTimeSpec(Qt::UTC);
    }
    BIO_free(memory);
  }

  return result;
}

QString Utils::oid(const int nid)
{
  ASN1_OBJECT *obj = OBJ_nid2obj(nid);
  if (!obj) {
    return QString();
  }

  size_t bufsize = 96, curBufsize = bufsize;
  char *name =(char *) malloc(bufsize);
  while (curBufsize < (bufsize = OBJ_obj2txt(name, bufsize, obj, 1)))
  {
    name =(char *) realloc(name, bufsize);
  }

  ASN1_OBJECT_free(obj);

  QString result(name);
  free(name);

  return result;
}

int Utils::registerOid(const QString &oid, const QString &shortName, const QString &longName)
{
  int resultNid;
  if ((resultNid = nid(oid)) == NID_undef)
  {
    resultNid = OBJ_create(oid.toLocal8Bit().constData(),
                     shortName.toLocal8Bit().constData(),
                     longName.toLocal8Bit().constData());
    X509V3_EXT_add_alias(resultNid, NID_netscape_comment);
  }
  else
  {
    int snNID = nid(shortName);
    int lnNID = nid(longName);

    if (!(snNID != NID_undef && resultNid == snNID && snNID == lnNID))
      return -1;
  }

  return resultNid;
}

int Utils::build_X509(X509 *x, quint64 serial, X509_NAME *subject, X509_NAME *issuer,
                      EVP_PKEY *key, const QDateTime &notBefore, const QDateTime &notAfter)
{
  if (!x || !subject || !issuer || !key)
    return 0;

  ASN1_INTEGER *sn = convert(serial);
  if (sn) {
    X509_set_serialNumber(x, sn);
    ASN1_INTEGER_free(sn);
  }
  else {
    return 0;
  }

  X509_set_version(x, 2);
  X509_set_subject_name(x, subject);
  X509_set_issuer_name(x, issuer);

  X509_set_pubkey(x, key);

  ASN1_TIME *tm = ASN1_TIME_new();
  ASN1_TIME_set(tm, notBefore.toUTC().toTime_t());
  X509_set_notBefore(x, tm);
  ASN1_TIME_set(tm, notAfter.toUTC().toTime_t());
  X509_set_notAfter(x, tm);
  ASN1_TIME_free(tm);

  return -1;
}
