/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include "../x509name.h"

#include <QtCore/QMultiMap>
#include <QtCore/QString>
#include <QtTest/QTest>

using namespace kCA_ossl;

/**
 * Test class for X509Name.
 * @short X509Name test class
 * @author Felix Tiede <infor@pc-tiede.de>
 */
class X509NameTest : public QObject
{
  Q_OBJECT
  private slots:
    /** Test creation of empty X509Name. */
    void createEmpty()
    {
      X509Name test;
      QCOMPARE(test.keys().length(), 0);
      QCOMPARE(test.toString(), QString("/"));
    }

    /** Test copy constructor. */
    void copy()
    {
      X509Name src("/C=AU/");
      X509Name dst(src);
      QCOMPARE(dst.keys().length(), src.keys().length());
      QCOMPARE(dst.toString(), src.toString());
    }

    /** Test assignment operator. */
    void assign()
    {
      X509Name src("/C=AU/");
      X509Name dst = src;
      QCOMPARE(dst.keys().length(), src.keys().length());
      QCOMPARE(dst.toString(), src.toString());
    }

    /** Test creation with a string. */
    void fromSimpleString()
    {
      QString source("/C=DE/ST=Hamburg/L=Hamburg/O=Example Ltd./OU=Test Department/CN=test/emailAddress=test@example.com/");
      X509Name test(source);

      QCOMPARE(test.toString().length(), source.length());

      // QMultiMap resorts keys in ascending order, thus a source string probably never matches itself
      QMultiMap<QString, QString> map = mapFromString(source);
      QCOMPARE(test.keys().length(), map.keys().length());

      QString compare = "/";
      foreach(QString key, map.uniqueKeys()) {
        foreach(QString value, map.values(key)) {
          compare += key + "=" + value + "/";
        }
      }
      QCOMPARE(test.toString(), compare);
    }

    /** Test creation from a QMultiMap. */
    void fromMultiMap()
    {
      QString source("/C=DE/ST=Hamburg/L=Hamburg/O=Example Ltd./OU=Test Department/CN=test/emailAddress=test@example.com/");
      QMultiMap<QString, QString> map = mapFromString(source);

      X509Name test(map);

      QCOMPARE(test.keys().length(), map.keys().length());

      QString msg;
      foreach(QString key, map.uniqueKeys()) {
        msg = key + ":" + test.values(key).join(",");
        QVERIFY2(test.values(key) == map.values(key), test.toString().toStdString().c_str());
      }
    }

    /** Test X509Name::addValue(). */
    void addValue()
    {
      X509Name test;
      test.addValues("emailAddress", QStringList() << "test2@example.com");

      QVERIFY2(test.values("emailAddress").contains("test2@example.com"), test.toString().toStdString().c_str());

      test.addValues("", "C", "Country", QStringList() << "DE");
      QVERIFY2(test.values("C").contains("DE"), test.toString().toStdString().c_str());
    }

    /** Test X509Name::addValues(). */
    void addMultiValue()
    {
      X509Name test;
      test.addValues("emailAddress", QStringList() << "test1@example.com" << "test2@example.com");

      QVERIFY2(test.values("emailAddress").contains("test1@example.com"), test.toString().toStdString().c_str());
      QVERIFY2(test.values("emailAddress").contains("test2@example.com"), test.toString().toStdString().c_str());
    }

  private:
    /** Create QMultiMap<QString, QString> from string. */
    inline QMultiMap<QString, QString> mapFromString(const QString &source)
    {
      QMultiMap<QString, QString> result;
      QStringList entryParts, values;

      foreach(QString entry, source.split("/", QString::SkipEmptyParts)) {
        entryParts = entry.split("=", QString::SkipEmptyParts);
        if (entryParts.size() > 1) {
          values = entryParts.mid(1);
          foreach(QString value, values) {
            result.insertMulti(entryParts[0], value);
          }
        }
      }

      return result;
    }
};

QTEST_MAIN(X509NameTest)
#include "x509nametest.moc"
