/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef _LIBKCAOSSL_REVOCATIONLIST_H
#define _LIBKCAOSSL_REVOCATIONLIST_H

#include "commons.h"

#include "kca_ossl_export.h"

#include <QtCore/QObject>
#include <QtCore/QDateTime>

template <typename T> class QList;
class QString;
class QUrl;

struct X509_crl_st;
typedef X509_crl_st X509_CRL;

/** Namespace for kca-local classes */
namespace kCA_ossl
{

class RevocationlistPrivate;
class Certificate;
class X509Name;
class X509Extension;

/**
 * Represents a certificate revocation list.
 *
 * @short certificate revocation list
 * @author Felix Tiede <info@pc-tiede.de>
 */
class _LIBKCAOSSL_EXPORT Revocationlist : public QObject
{
  friend class RevocationlistPrivate;
  friend class Certificate;

  Q_OBJECT
  public:
    /** Creates an empty revocation list. */
    Revocationlist(QObject *parent = 0);
    /** Copy constructor. */
    Revocationlist(const Revocationlist &other, QObject *parent = 0);
    /**
     * Creates a revocation list from issuer and number.
     * Extensions and initial entries may be given as parameters
     * but cannot be removed later.
     *
     * @param issuer The issuer of the list.
     * @param number The sequence number of the list.
     * @param parent Owner object of this instance, default 0.
     */
    Revocationlist(const Certificate &issuer, quint64 number, QObject* parent = 0);
    ~Revocationlist();

    /** Get issuer of certificate revocation list. */
    const X509Name &issuer() const;
    /** Get issuer hash value of certificate revocation list. */
    quint32 issuerHash() const;
    /** Get serial number of certificate revocation list. */
    quint64 serial() const;
    /** Get timestamp of this revocation list. For unsigned list, it's QDateTime(). */
    const QDateTime &thisUpdate() const;
    /** Get timestamp of next update of revocation list. */
    const QDateTime &nextUpdate() const;
    /** Set timestamp of next update of revocation lsit. */
    void setNextUpdate(const QDateTime &timestamp);

    /** Get digest used to sign revocation list. */
    Digest digest() const;

    /** Get list of extensions of revocation list. */
    const QList< const X509Extension* > &extensions() const;

    /**
     * Add an extension to revocation list. If an extension with the same
     * OID is already stored, that extension is overwritten.
     *
     * @param extension Extension to add to revocation list.
     */
    void addExtension(const X509Extension &extension);

    /** Set distribution point extension of revocation list. */
    void setDistributionPoint(const QUrl &url);

    /**
     * Get a list of certificates in revocation list. The list is composed
     * of the certificate's serial numbers.
     */
    const QList<quint64> entries() const;
    /**
     * Adds a certificate to revocation list.
     * If the certificate is already in the list, the operation is silently
     * ignored unless @p reason is RevocationReasons::removeFromCRL in which case
     * the entry is removed from revocation list.
     * If issuer of certificate and revocation list do not match, the operation is
     * not performed.
     *
     * @param certificate Certificate to be revoked.
     * @param reason Reason of entry to revocation list.
     * @param timestamp Date and time of revocation of certificate, defaults to now.
     */
    void addEntry(const Certificate &certificate, RevocationReason reason,
                  const QDateTime &timestamp = QDateTime::currentDateTime());

    /**
     * Write revocation list to PEM formatted file.
     * The file will be overwritten.
     *
     * @param filename File to write the revocation list to.
     * @return true if successful, otherwise false.
     */
    bool toPemFile(const QString &filename) const;
    /**
     * Write revocation list to DER formatted file.
     * The file will be overwritten.
     *
     * @param filename File to write the revocation list to.
     * @return true if successful, otherwise false
     */
    bool toDerFile(const QString &filename) const;

    /** Assignment operator. */
    Revocationlist& operator=(const Revocationlist &other);

  protected:
    /**
     * Create revocation list from OpenSSL's internal crl representation,
     * X509_CRL.
     *
     * @param source X509_CRL struct to be parsed.
     * @param parent Owner object of this instance, default 0.
     */
    Revocationlist(X509_CRL *source, QObject *parent = 0);

    /** Set timestamp of this revocation list. Use prior to signing. */
    void setThisUpdate(const QDateTime &timestamp);

    /** Gets OpenSSL's internal representation of revocation list. */
    X509_CRL *internal() const;

  private:
    /** Local value storage, protect binary compatibility. */
    RevocationlistPrivate * const d;
};

}

#endif // _LIBKCAOSSL_REVOCATIONLIST_H
