/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "revocationlist.h"

#include "utils.h"
#include "certificate.h"
#include "x509extension.h"
#include "x509name.h"

#include "kca_ossl_export.h"

#include <QtCore/QDebug>
#include <QtCore/QObject>
#include <QtCore/QDateTime>
#include <QtCore/QFile>
#include <QtCore/QList>
#include <QtCore/QMap>
#include <QtCore/QString>
#include <QtCore/QUrl>

#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

namespace kCA_ossl
{

/**
 * Represents a single entry of revocation list
 *
 * @short certificate revocation entry
 * @author Felix Tiede <info@pc-tiede.de>
 */
class RevocationEntry
{
  public:
    /** Create an entry with given values. */
    RevocationEntry(const quint64 serial, const RevocationReason reason,
                    const QDateTime &timestamp) :
        serial(serial),
        timestamp(timestamp.toUTC()),
        reason(reason)
    {}
    /** Default destructor, no action performed. */
    ~RevocationEntry() {}

    /** Compare two entries by their timestamp, which is how they are sorted. */
    bool operator< (const RevocationEntry &other)
    {
      return timestamp < other.timestamp;
    }
    /** Compare two entries by their timestamp, which is how they are sorted. */
    bool operator> (const RevocationEntry &other)
    {
      return timestamp > other.timestamp;
    }

    /** Revoked certificate's serial number. */
    quint64 serial;
    /** Timestamp of revocation. */
    QDateTime timestamp;
    /** Reason of revocation. */
    RevocationReason reason;
};

/**
 * Hidden class to store class Revocationlist's instance values.
 * Purpose is to ensure binary compatibility.
 *
 * @short Revocationlist instance value storage.
 * @author Felix Tiede <info@pc-tiede.de>
 */
class RevocationlistPrivate
{
  public:
    /** Create an empty revocation list. */
    RevocationlistPrivate() :
        serial(0),
        internal(0)
    {}
    /** Initializes instance storage with values from source instance. */
    RevocationlistPrivate(const RevocationlistPrivate *other) :
        serial(other->serial),
        issuer(other->issuer),
        issuerKeyId(other->issuerKeyId),
        thisUpdate(other->thisUpdate),
        nextUpdate(other->nextUpdate),
        list(other->list),
        extensionList(other->extensionList),
        internal(0)
    {
      if (other->internal)
        internal = X509_CRL_dup(other->internal);
    }
    /** Clean up our mess in memory. */
    ~RevocationlistPrivate()
    {
      if (internal) {
        X509_CRL_free(internal);
      }

      list.clear();
      extensionList.clear();
    }

    /** Initialise internal to a new X509_CRL struct and set up timestamps for X509_CRL_dup(). */
    void initInternal()
    {
      if (!internal) {
        internal = X509_CRL_new();
        X509_CRL_set_version(internal, 1);

        // Feed a default timestamp, otherwise X509_CRL_dup() fails.
        ASN1_TIME *tm = ASN1_TIME_new();
        if (tm) {
          ASN1_TIME_set(tm, QDateTime::currentDateTime().toUTC().toTime_t());
          X509_CRL_set_lastUpdate(internal, tm);
          X509_CRL_set_nextUpdate(internal, tm);
          ASN1_TIME_free(tm);
        }
      }
    }

    /** Parses a X509_CRL struct to our internal data storage. */
    void parseValues()
    {
      issuer = X509Name(internal->crl->issuer);
      thisUpdate = Utils::convert(internal->crl->lastUpdate);
      nextUpdate = Utils::convert(internal->crl->nextUpdate);

      // Load revoked certificates
      BIGNUM *revSerial;
      X509_REVOKED *rev;
      X509_EXTENSION *revExt;
      RevocationEntry *entry;
      QDateTime revTime;
      RevocationReason revReason = Unspecified;
      if (internal->crl->revoked) {
        for (int i=0; i<internal->crl->revoked->stack.num; ++i) {
          rev =(X509_REVOKED *) internal->crl->revoked->stack.data[i];
          revTime = Utils::convert(rev->revocationDate);
          revSerial = ASN1_INTEGER_to_BN(rev->serialNumber, NULL);
          //TODO: Load reason to RevocationEntry
          revExt = X509_REVOKED_get_ext(rev, X509_REVOKED_get_ext_by_NID(rev, NID_crl_reason, -1));
          entry = new RevocationEntry(QString::fromLocal8Bit(BN_bn2hex(revSerial)).toULongLong(NULL, 16),
                                      revReason, Utils::convert(rev->revocationDate));
          BN_free(revSerial);
          list.insert(revTime, entry);
        }
      }

      // Load extensions
      X509Extension *ext;
      for (int i=0; i<X509_CRL_get_ext_count(internal); ++i) {
        ext = new X509Extension(X509_CRL_get_ext(internal, i));
        if (ext->isValid()) {
          extensionList << ext;
          switch (ext->getNid()) {
            case NID_crl_number:
              serial = ext->value().toULongLong(0, 10);
              break;
            case NID_authority_key_identifier:
              foreach (QString value, ext->value().split('\n')) {
                if (value.startsWith("keyid:")) {
                  issuerKeyId = value.right(value.size()-6);
                  break;
                }
              }
              break;
          }
        }
      }
    }

    RevocationlistPrivate& operator=(const RevocationlistPrivate& other)
    {
      serial = other.serial;
      issuer = other.issuer;
      issuerKeyId = other.issuerKeyId;
      thisUpdate = other.thisUpdate;
      nextUpdate = other.nextUpdate;
      list = other.list;
      extensionList = other.extensionList;

      if (other.internal)
        internal = X509_CRL_dup(other.internal);

      return *this;
    }

    /** Serial number of revocation list. */
    quint64 serial;

    /** Issuer of revocation list. */
    X509Name issuer;

    /** Issuer's key id of revocation list. */
    QString issuerKeyId;

    /** Timestamp of this update, set at signing time. */
    QDateTime thisUpdate;

    /** Timestamp when this revocation list is outdated. */
    QDateTime nextUpdate;

    /** List of revoked certificates, ordered by the time when they were revoked. */
    QMap< QDateTime, const RevocationEntry* > list;

    /** List of extensions for revocation list. */
    QList< const X509Extension* > extensionList;

    /** OpenSSL's internal representation of revocation list. */
    X509_CRL *internal;
};

} // End namespace kCA_ossl

using namespace kCA_ossl;

Revocationlist::Revocationlist(QObject *parent) : QObject(parent), d(new RevocationlistPrivate)
{}

Revocationlist::Revocationlist(const Revocationlist &other, QObject *parent) :
    QObject(parent),
    d(new RevocationlistPrivate(other.d))
{}

Revocationlist::Revocationlist(const Certificate &issuer, quint64 number, QObject *parent) :
    QObject(parent),
    d(new RevocationlistPrivate)
{
  d->serial = number;
  d->issuer = issuer.subject();
  d->issuerKeyId = issuer.keyId();

  d->internal = X509_CRL_new();
  if (d->internal) {
    X509_CRL_set_version(d->internal, 1);
    X509_CRL_set_issuer_name(d->internal, issuer.subject().internal());
    ASN1_INTEGER *crlnumber = Utils::convert(number);
    if (crlnumber) {
      X509_CRL_add1_ext_i2d(d->internal, NID_crl_number, crlnumber, 0, 0);
      ASN1_INTEGER_free(crlnumber);
    }

    // Feed a default timestamp, otherwise X509_CRL_dup() fails.
    ASN1_TIME *tm = ASN1_TIME_new();
    if (tm) {
      ASN1_TIME_set(tm, QDateTime::currentDateTime().toUTC().toTime_t());
      X509_CRL_set_lastUpdate(d->internal, tm);
      X509_CRL_set_nextUpdate(d->internal, tm);
      ASN1_TIME_free(tm);
    }
  }
}

Revocationlist::Revocationlist(X509_CRL *source, QObject *parent) :
    QObject(parent),
    d(new RevocationlistPrivate)
{
  if (!source)
    return;

  d->internal = X509_CRL_dup(source);
  if (d->internal) {
    d->parseValues();
  }
}

Revocationlist::~Revocationlist()
{
  delete d;
}

const X509Name &Revocationlist::issuer() const
{
  return d->issuer;
}

quint32 Revocationlist::issuerHash() const
{
  if (d->internal->crl && d->internal->crl->issuer)
    return X509_NAME_hash(d->internal->crl->issuer);

  return 0;
}

quint64 Revocationlist::serial() const
{
  return d->serial;
}

const QDateTime &Revocationlist::thisUpdate() const
{
  return d->thisUpdate;
}

const QDateTime &Revocationlist::nextUpdate() const
{
  return d->nextUpdate;
}

void Revocationlist::setNextUpdate(const QDateTime &timestamp)
{
  if (!timestamp.isValid() || timestamp < QDateTime::currentDateTime()) {
    return;
  }

  d->nextUpdate = timestamp;

  d->initInternal();
  if (d->internal) {
    ASN1_TIME *ts = ASN1_TIME_new();
    if (ts) {
      ASN1_TIME_set(ts, timestamp.toUTC().toTime_t());
      X509_CRL_set_nextUpdate(d->internal, ts);
      ASN1_TIME_free(ts);
    }
  }
}

Digest Revocationlist::digest() const
{
  if (!d->internal) {
    return kCA_ossl::SHA1;
  }

  int nid = OBJ_obj2nid(d->internal->sig_alg->algorithm);
  switch (nid) {
    case 119:
      return kCA_ossl::RIPEMD160;
      break;
    case 65:
      return kCA_ossl::SHA1;
      break;
    case 668:
      return kCA_ossl::SHA256;
      break;
    case 669:
      return kCA_ossl::SHA384;
      break;
    case 670:
      return kCA_ossl::SHA512;
      break;
  }

  return kCA_ossl::SHA1;
}

const QList< const X509Extension* > &Revocationlist::extensions() const
{
  return d->extensionList;
}

void Revocationlist::addExtension(const X509Extension &extension)
{
  d->initInternal();
  X509_EXTENSION *extData = extension.internal();
  if (d->internal && extData) {
    X509_CRL_add_ext(d->internal, extData, -1);
    d->extensionList << new X509Extension(extension);
  }
}

void Revocationlist::setDistributionPoint(const QUrl &url)
{
  d->initInternal();

  QByteArray encoded_url = url.toEncoded(QUrl::RemoveUserInfo);
  if (encoded_url.isEmpty()) {
    return;
  }
  encoded_url = QByteArray("fullname=URI:") + encoded_url;

  X509V3_CTX v3_context;
  X509V3_set_ctx_nodb(&v3_context);
  X509V3_set_ctx(&v3_context, NULL, NULL, NULL, d->internal, 0);

  X509V3_EXT_add_alias(NID_issuing_distribution_point, NID_netscape_comment);
  X509_EXTENSION *extension = X509V3_EXT_conf_nid(NULL, &v3_context, NID_issuing_distribution_point,
                                                  (char *) encoded_url.constData());
  if (extension) {
    X509_EXTENSION_set_critical(extension, 1);
    X509_CRL_add_ext(d->internal, extension, -1);
    X509_EXTENSION_free(extension);
  }
}

const QList< quint64 > Revocationlist::entries() const
{
  QList< quint64 > result;
  foreach(const RevocationEntry *entry, d->list.values()) {
    result << entry->serial;
  }

  return result;
}

void Revocationlist::addEntry(const Certificate &certificate, RevocationReason reason,
                              const QDateTime &timestamp)
{
  if (d->issuerKeyId.length()>0 && certificate.issuerKeyId().length()>0) {
    // If existent, key id's of issuers of revocation list and certificate must match.
    if (d->issuerKeyId != certificate.issuerKeyId()) {
      return;
    }
  }
  // Also both issuer's subject names must match
  if (d->issuer.toString() != certificate.issuer().toString()) {
    return;
  }

  quint64 serial = certificate.serial();
  RevocationEntry *entry = new RevocationEntry(serial, reason, timestamp);
  X509_REVOKED *crlEntry = X509_REVOKED_new();
  if (!crlEntry) {
    return;
  }

  d->initInternal();

  ASN1_INTEGER *crlEntrySn = Utils::convert(entry->serial);
  if (crlEntrySn) {
    X509_REVOKED_set_serialNumber(crlEntry, crlEntrySn);
    ASN1_INTEGER_free(crlEntrySn);
  }
  ASN1_TIME *crlEntryTs = ASN1_TIME_new();
  if (crlEntryTs) {
    ASN1_TIME_set(crlEntryTs, entry->timestamp.toTime_t());
    X509_REVOKED_set_revocationDate(crlEntry, crlEntryTs);
    ASN1_TIME_free(crlEntryTs);
  }

  ASN1_ENUMERATED *tmpReason = ASN1_ENUMERATED_new();
  ASN1_ENUMERATED_set(tmpReason, reason);
  X509_REVOKED_add1_ext_i2d(crlEntry, NID_crl_reason, tmpReason, 0, 0);
  ASN1_ENUMERATED_free(tmpReason);

  if (reason == RemoveFromCRL) {
    delete entry;

    bool move = false;
    X509_REVOKED *item;
    for (int i=0; i<d->internal->crl->revoked->stack.num; ++i) {
      item =(X509_REVOKED *) d->internal->crl->revoked->stack.data[i];
      if (!ASN1_INTEGER_cmp(item->serialNumber, crlEntry->serialNumber)) {
        X509_REVOKED_free((X509_REVOKED *) d->internal->crl->revoked->stack.data[i]);
        d->internal->crl->revoked->stack.num--;
        move = true;
      }
      if (move) {
        d->internal->crl->revoked->stack.data[i] = d->internal->crl->revoked->stack.data[i+1];
      }
    }
    foreach (const RevocationEntry *entry, d->list.values()) {
      if (entry->serial == serial) {
        d->list.remove(entry->timestamp);
        break;
      }
    }
    return;
  }

  X509_CRL_add0_revoked(d->internal, crlEntry);

  d->list.insert(timestamp, entry);
  delete entry;
}

bool Revocationlist::toPemFile(const QString &filename) const
{
  if (!d->internal) {
    return false;
  }

  QFile crlfile(filename);
  if (!crlfile.open(QFile::WriteOnly)) {
    return false;
  }

  bool result = false;

  FILE *out = fdopen(crlfile.handle(), "w");
  if (out) {
    if (!d->thisUpdate.isValid()) {
      ASN1_TIME *ts = ASN1_TIME_new();
      ASN1_TIME_set(ts, QDateTime::currentDateTime().toUTC().toTime_t());
      X509_CRL_set_lastUpdate(d->internal, ts);
      ASN1_TIME_free(ts);
    }
    if (PEM_write_X509_CRL(out, d->internal)) {
      result = true;
    }

    fclose(out);
  }
  crlfile.close();

  return result;
}

bool Revocationlist::toDerFile(const QString &filename) const
{
  if (!d->internal) {
    return false;
  }

  QFile crlfile(filename);
  if (!crlfile.open(QFile::WriteOnly)) {
    return false;
  }

  bool result = false;

  BIO *b_out = BIO_new_fd(crlfile.handle(), BIO_CLOSE);
  if (b_out) {
    if (!d->thisUpdate.isValid()) {
      ASN1_TIME *ts = ASN1_TIME_new();
      ASN1_TIME_set(ts, QDateTime::currentDateTime().toUTC().toTime_t());
      X509_CRL_set_lastUpdate(d->internal, ts);
      ASN1_TIME_free(ts);
    }
    if (i2d_X509_CRL_bio(b_out, d->internal)) {
      result = true;
    }

    BIO_free(b_out);
  }
  crlfile.close();

  return result;
}

void Revocationlist::setThisUpdate(const QDateTime &timestamp)
{
  if (!timestamp.isValid() || (d->nextUpdate.isValid() &&
      timestamp > d->nextUpdate)) {
    return;
  }

  d->thisUpdate = timestamp;

  d->initInternal();
  if (d->internal) {
    ASN1_TIME *ts = ASN1_TIME_new();
    if (ts) {
      ASN1_TIME_set(ts, timestamp.toUTC().toTime_t());
      X509_CRL_set_lastUpdate(d->internal, ts);
      ASN1_TIME_free(ts);
    }
  }
}

X509_CRL *Revocationlist::internal() const
{
  if (d->internal)
    return X509_CRL_dup(d->internal);

  return 0;
}

Revocationlist& Revocationlist::operator=(const kCA_ossl::Revocationlist& other)
{
  *d = *(other.d);
  return *this;
}
