/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009, 2010 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "request.h"

#include "certificate.h"
#include "utils.h"
#include "x509name.h"
#include "x509attribute.h"
#include "x509extension.h"

#include <QtCore/QDebug>
#include <QtCore/QFile>
#include <QtCore/QList>

#include <openssl/asn1.h>
#include <openssl/bn.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/rand.h>
#include <openssl/safestack.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#define MIN_KEY_LENGTH 2048

namespace kCA_ossl
{

/**
 * Hidden class to store class Request's instance values.
 * Purpose is to ensure binary compatibility.
 *
 * @short Request instance value storage.
 * @author Felix Tiede <info@pc-tiede.de>
 */
class RequestPrivate
{
  public:
    /** Initialize a few local pointers to null. */
    RequestPrivate(const QString &filename = QString()) :
        valid(false),
        sourcefile(0),
        keyfile(0),
        internal(0),
        privateKey(0),
        attributeList(QList< const X509Attribute * >()),
        extensionList(QList< const X509Extension * >())
    {
      if (!filename.isEmpty())
        valid = readfile(filename);
    }

    /** Copy constructor. */
    RequestPrivate(const RequestPrivate *other) :
        valid(other->valid),
        sourcefile(new QFile(other->sourcefile->fileName())),
        keyfile(new QFile(other->keyfile->fileName())),
        internal(0),
        privateKey(0),
        attributeList(other->attributeList),
        extensionList(other->extensionList)
    {
      if (other->internal)
        internal = X509_REQ_dup(other->internal);

      if (other->privateKey) {
        BIO *mem = BIO_new(BIO_s_mem());
        i2d_PrivateKey_bio(mem, other->privateKey);
        d2i_PrivateKey_bio(mem, &privateKey);
        BIO_free(mem);
      }
    }

    /** Clean up our own memory. */
    ~RequestPrivate()
    {
      valid = false;
      attributeList.clear();
      extensionList.clear();
      if (sourcefile) {
        sourcefile->close();
      }
      if (keyfile) {
        keyfile->close();
      }
      if (internal) {
        X509_REQ_free(internal);
      }
      if (privateKey) {
        EVP_PKEY_free(privateKey);
      }
    }

    /** Try to read file into memory. */
    inline bool readfile(const QString &filename)
    {
      sourcefile = new QFile(filename);
      if (!sourcefile->exists()) {
        return false;
      }

      sourcefile->open(QFile::ReadOnly);

      FILE* fh = fdopen(sourcefile->handle(), "r");
      if (fh == NULL) {
        sourcefile->close();
        return false;
      }

      internal = PEM_read_X509_REQ(fh, NULL, NULL,(void *) "");
      if (!internal) {
        rewind(fh);
        internal = d2i_X509_REQ_fp(fh, NULL);
      }

      fclose(fh);
      sourcefile->close();

      // We could not read whatever we were given
      if (!internal) {
        return false;
      }

      X509Attribute *attr;
      for (int i=0; i<internal->req_info->attributes->stack.num; ++i) {
        attr = new X509Attribute((X509_ATTRIBUTE *) internal->req_info->attributes->stack.data[i]);
        if (attr->requestedExtension()) {
          continue;
        }
        attributeList << attr;
      }

      STACK_OF(X509_EXTENSION) *exts = X509_REQ_get_extensions(internal);
      if (exts) {
        for (int i=0; i<exts->stack.num; ++i) {
          extensionList << new X509Extension((X509_EXTENSION *) exts->stack.data[i]);
        }
      }

      return true;
    }

    /** Generate a new keypair. */
    inline bool newKeypair(const X509Name &subject, int size, const Digest digest,
                           const Request *caller)
    {
      // Invalid until creation of request with new keys was successwul
      valid = false;

      if (size < MIN_KEY_LENGTH) {
        size = MIN_KEY_LENGTH;
      }

      // Create data structure if it doesn't exist and do not continue if failed
      if (!internal) {
        internal = X509_REQ_new();
      }
      if (!internal) {
        return false;
      }

      X509_REQ_set_version(internal, 0L);
      X509_REQ_set_subject_name(internal, subject.internal());
      foreach(const X509Attribute *attr, attributeList) {
        if (X509_ATTRIBUTE *a = attr->internal()) {
          X509_REQ_add1_attr(internal, a);
          X509_ATTRIBUTE_free(a);
        }
      }

      RAND_load_file(Utils::random_source, size);
      BN_GENCB cb;
      BN_GENCB_set(&cb, &process,(void *) caller);

      privateKey = EVP_PKEY_new();
      RSA *rsa = RSA_new();
      BIGNUM *bn = BN_new();

      // Do the work - may last a while, so call the callback
      if (!rsa || !bn || !BN_set_word(bn, 0x10001) ||
          !RSA_generate_key_ex(rsa, size, bn, &cb) ||
          !EVP_PKEY_assign_RSA(privateKey, rsa)) {
        BN_free(bn);
        RSA_free(rsa);
        return false;
      }

      BN_free(bn);

      RAND_write_file(Utils::random_source);

      X509_REQ_set_pubkey(internal, privateKey);
      X509_REQ_sign(internal, privateKey, Utils::load_digest(digest));

      // Creation was successful, we are valid from now on
      return true;
    }

    /** Instance is valid and may be used or queried for values. */
    bool valid;

    /** File where this instance was read from. */
    QFile *sourcefile;
    /** File where private key is located. */
    QFile *keyfile;

    /** OpenSSL's internal representation of request. */
    X509_REQ *internal;
    /** Private key data structure, filled by Request::generate() or Request::loadPrivateKey(). */
    EVP_PKEY *privateKey;

    /** List of attributes of request. */
    QList< const X509Attribute* > attributeList;
    /** List of requested extensions in request. */
    QList< const X509Extension* > extensionList;

    /**
     * Callback to show process of generating big numbers
     * (aka. cryptographic keys).
     * @see http://www.openssl.org/docs/crypto/BN_generate_prime.html
     * @see http://www.openssl.org/docs/crypto/RSA_generate_key.html
     *
     * @param p state of prime generation
     * <!--@param n number of prime generated-->
     * @param cb callback to use to display information
     */
    static int process(int p, int, BN_GENCB *cb)
    {
      Request *instance =(Request *) cb->arg;
      switch(p) {
        case 0:
          emit instance->keyBuildProcess(QChar('.'));
          break;
        case 1:
          emit instance->keyBuildProcess(QChar('+'));
          break;
        case 2:
        default:
          emit instance->keyBuildProcess(QChar('*'));
          break;
        case 3:
          emit instance->keyBuildProcess(QChar('\n'));
          break;
      }

      return 1;
    }
};

} // End namespace kCA_ossl

using namespace kCA_ossl;

Request::Request(QObject* parent) : QObject(parent), d(new RequestPrivate)
{}

Request::Request(const Request& other) :
    QObject(other.parent()),
    d(new RequestPrivate(other.d))
{}

Request::Request(const QString &filename, QObject *parent) :
    QObject(parent),
    d(new RequestPrivate(filename))
{}

Request::~Request()
{
  delete d;
}

bool Request::isValid() const
{
  return d->valid;
}

int Request::keyBits() const
{
  if (d->valid) {
    return (d->internal->req_info->pubkey->public_key->length-14)*8;
  }

  return -1;
}

const X509Name Request::subject() const
{
  if (d->valid) {
    return X509Name(d->internal->req_info->subject);
  }

  return X509Name();
}

const QList< const X509Attribute* > &Request::attributes() const
{
  return d->attributeList;
}

const QList< const X509Extension* > &Request::extensions() const
{
  return d->extensionList;
}

X509_REQ *Request::internal() const
{
  return d->internal;
}

void Request::toPemFile(const QString& filename, WriteMode mode,
                        const QString& keyfile, const QString& password) const
{
  d->sourcefile = new QFile(filename);
  d->sourcefile->open(QFile::WriteOnly);
  d->sourcefile->setPermissions(QFile::ReadOwner | QFile::ReadGroup | QFile::ReadOwner | QFile::WriteOwner);
  FILE *out = fdopen(d->sourcefile->handle(), "w");
  PEM_write_X509_REQ(out, d->internal);
  fclose(out);
  d->sourcefile->close();

  // No private key, we're done
  if (!d->privateKey) {
    return;
  }

  // Whatever the keyfile, we don't need it in case of samefile == true
  switch (mode) {
    default:
    case NoPrivateKey:
      return;
      break;
    case SameFile:
      d->keyfile = d->sourcefile;
      break;
    case OtherFile:
      d->keyfile = new QFile(keyfile);
      break;
  }
  keyToPemFile(d->keyfile->fileName(), password);
}

void Request::keyToPemFile(const QString& filename, const QString& password) const
{
  if (filename.length() == 0) {
    return;
  }
  if (d->keyfile) {
    if (filename != d->keyfile->fileName()) {
      delete d->keyfile;
      d->keyfile = new QFile(filename);
    }
  }
  else {
    d->keyfile = new QFile(filename);
  }

  RSA *rsa = EVP_PKEY_get1_RSA(d->privateKey);
  if (!rsa) {
    return;
  }

  d->keyfile->open(QFile::Append);
  d->keyfile->setPermissions(QFile::ReadOwner | QFile::WriteOwner);
  FILE *out = fdopen(d->keyfile->handle(), "a");
  if (password.length() == 0) {
    PEM_write_RSAPrivateKey(out, rsa, NULL, NULL, 0, 0, NULL);
  }
  else {
    PEM_write_RSAPrivateKey(out, rsa, EVP_aes_256_cbc(), NULL, 0, 0, password.toUtf8().data());
  }
  fclose(out);
  d->keyfile->close();

  RSA_free(rsa);
}

const Certificate Request::selfsign(quint64 serial, const Digest digest,
                                    const QDateTime& notBefore, const QDateTime& notAfter,
                                    const QList< const X509Extension* >& extensions,
                                    const QString& keyFile, const QString &password) const
{
  if (!Utils::OSSL_ciphers_loaded)
  {
    OpenSSL_add_all_ciphers();
    Utils::OSSL_ciphers_loaded = true;
  }

  // Load private key
  if (!d->privateKey) {
    d->privateKey = EVP_PKEY_new();
    d->keyfile = new QFile(keyFile);
    if (!d->keyfile->exists()) {
      return Certificate();
    }
    d->keyfile->open(QFile::ReadOnly);
    FILE *keyfile = fdopen(d->keyfile->handle(), "r");
    if (!PEM_read_PrivateKey(keyfile, &d->privateKey, NULL,(void *) password.toAscii().data()))
    {
      rewind(keyfile);
      RSA *rsa = d2i_RSAPrivateKey_fp(keyfile, NULL);
      EVP_PKEY_assign_RSA(d->privateKey, rsa);
      RSA_free(rsa);
    }
    fclose(keyfile);
    d->keyfile->close();
  }

  // Check if private key matches request's public_key
  if (!X509_REQ_check_private_key(d->internal, d->privateKey)) {
    EVP_PKEY_free(d->privateKey);
    return Certificate();
  }

  // Create certificate
  X509 *newcert = X509_new();
  EVP_PKEY *lkey = X509_REQ_get_pubkey(d->internal);
  EVP_PKEY_copy_parameters(lkey, d->privateKey);

  if (!Utils::build_X509(newcert, serial, d->internal->req_info->subject,
        d->internal->req_info->subject, lkey, notBefore, notAfter))
  {
    EVP_PKEY_free(lkey);
    EVP_PKEY_free(d->privateKey);
    X509_free(newcert);
    return Certificate();
  }

  EVP_PKEY_free(lkey);

  X509V3_CTX v3_context;
  X509V3_set_ctx_nodb(&v3_context);
  X509V3_set_ctx(&v3_context, newcert, newcert, NULL, NULL, 0);

  QList< const X509Extension* > add_ext = extensions;
  if (extensions.count() == 0) {
    add_ext = Utils::emailCertConstraints;
    foreach(QString value, subject().values("CN")) {
      if (value.endsWith("ca", Qt::CaseInsensitive) ||
          value.endsWith("certification authority", Qt::CaseInsensitive))
      {
        add_ext = Utils::CACertConstraints;
        break;
      }
    }
  }

  foreach(const X509Extension *ext, add_ext) {
    X509_EXTENSION *x = X509V3_EXT_conf_nid(NULL, &v3_context, ext->getNid(),(char *) ext->value().constData());
    if (x) {
      X509_EXTENSION_set_critical(x, (ext->isCritical() ? 1 : 0));
      X509_add_ext(newcert, x, -1);
      X509_EXTENSION_free(x);
    }
  }

  int result = X509_sign(newcert, d->privateKey, Utils::load_digest(digest));

  if (result) {
    BIO *memory = BIO_new(BIO_s_mem());
    i2d_PrivateKey_bio(memory, d->privateKey);
    BUF_MEM *privateKeyData;
    BIO_get_mem_ptr(memory, &privateKeyData);
    Certificate cert(newcert, QByteArray(privateKeyData->data, privateKeyData->length), parent());
    BIO_free(memory);
    X509_free(newcert);
    return cert;
  }

  X509_free(newcert);
  return Certificate();
}

Request Request::generate(const X509Name &subject, int size, const Digest digest,
                          const QList< const X509Attribute* > &attributes)
{
  Request result;
  result.d->attributeList = attributes;
  result.d->valid = result.d->newKeypair(subject, size, digest, &result);
  return result;
}
