/*
    kCA, a KDE Certification Authority management tool
    Copyright (C) 2009 - 2011 Felix Tiede <info@pc-tiede.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "certificate.h"

#include "utils.h"
#include "request.h"
#include "revocationlist.h"
#include "x509name.h"
#include "x509extension.h"

#include <QtCore/QDebug>
#include <QtCore/QDateTime>
#include <QtCore/QByteArray>
#include <QtCore/QFile>
#include <QtCore/QList>
#include <QtCore/QLocale>
#include <QtCore/QString>
#include <QtCore/QStringList>

#ifdef USE_QTNETWORK
  #include <QtNetwork/QSsl>
  #include <QtNetwork/QSslCertificate>
#endif // USE_QTNETWORK

#include <openssl/asn1.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/objects.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#ifdef _MSC_VER
  #define fdopen(fd, mode) _fdopen(fd, mode)
#endif // _MSC_VER

namespace kCA_ossl
{

/**
 * Hidden class to store class Certificate's instance values.
 * Purpose is to ensure binary compatibility.
 *
 * @short Certificate instance value storage.
 * @author Felix Tiede <info@pc-tiede.de>
 */
class CertificatePrivate
{
  public:
    /** Initializes instance storage with clean values. */
    CertificatePrivate() :
        valid(false),
        internal(0),
        serial(0),
        isCA(false),
        privateKey(0),
        notBefore(QDateTime()),
        notAfter(QDateTime()),
        keyId(QString()),
        issuerKeyId(QString()),
        extensionList(QList< const X509Extension * >())
    {}
    /** Initializes instance storage with values from source instance. */
    CertificatePrivate(const CertificatePrivate *other) :
        valid(other->valid),
        internal(0),
        serial(other->serial),
        isCA(other->isCA),
        privateKey(0),
        notBefore(other->notBefore),
        notAfter(other->notAfter),
        keyId(other->keyId),
        issuerKeyId(other->issuerKeyId),
        extensionList(other->extensionList)
    {
      if (valid)
        internal = X509_dup(other->internal);

      if (other->privateKey) {
        BIO *mem = BIO_new(BIO_s_mem());
        i2d_PrivateKey_bio(mem, other->privateKey);
        d2i_PrivateKey_bio(mem, &privateKey);
        BIO_free(mem);
      }
    }
    /** Free used memory. */
    ~CertificatePrivate()
    {
      valid = false;
      extensionList.clear();
      if (internal) {
        X509_free(internal);
      }
      if (privateKey) {
        EVP_PKEY_free(privateKey);
      }
    }

    /**
     * Parses extensions from OpenSSL representation into a QList.
     * Sets the isCA flag if basic constraints extension is found and
     * declares certificate to be a CA certificate.
     */
    inline void parseValues()
    {
      BIGNUM *serial = ASN1_INTEGER_to_BN(internal->cert_info->serialNumber, NULL);
      this->serial = QString::fromLocal8Bit(BN_bn2hex(serial)).toULongLong(NULL, 16);
      BN_free(serial);

      int nid, count = X509_get_ext_count(internal);
      X509Extension *ext;
      for (int i=0; i<count; ++i) {
        ext = new X509Extension(X509_get_ext(internal, i));
        if (ext->isValid()) {
          nid = ext->getNid();
          extensionList << ext;
          switch (nid) {
            case NID_basic_constraints:
              // Find out if certificate is supposed to be a CA
              if (ext->isCritical() && ext->value().toUpper() == "CA:TRUE") {
                  isCA = true;
              }
              break;
            // Load key ID and issuer key ID
            case NID_subject_key_identifier:
              keyId = ext->value();
              break;
            case NID_authority_key_identifier:
              foreach (QString value, ext->value().split('\n')) {
                if (value.startsWith("keyid:")) {
                  issuerKeyId = value.right(value.size()-6);
                  break;
                }
              }
              break;
          }
        }
        else  {
          delete ext;
        }
      }

      notBefore = Utils::convert(internal->cert_info->validity->notBefore);
      notAfter  = Utils::convert(internal->cert_info->validity->notAfter);
    }

    CertificatePrivate& operator=(const CertificatePrivate& other)
    {
      valid = other.valid;
      serial = other.serial;
      isCA = other.isCA;
      notBefore = other.notBefore;
      notAfter = other.notAfter;
      keyId = other.keyId;
      issuerKeyId = other.issuerKeyId;
      extensionList = other.extensionList;

      if (valid)
        internal = X509_dup(other.internal);

      if (other.privateKey) {
        BIO *mem = BIO_new(BIO_s_mem());
        i2d_PrivateKey_bio(mem, other.privateKey);
        d2i_PrivateKey_bio(mem, &privateKey);
        BIO_free(mem);
      }

      return *this;
    }

    /** Instance is valid and may be used or queried for values. */
    bool valid;
    /** OpenSSL's internal representation of certificate. */
    X509 *internal;

    /** Serial number of certificate. */
    quint64 serial;

    /** Decrlares instance to be a CA and allows execution of Certificate::signRequest(). */
    bool isCA;
    /** Private key data structure, may be filled and cleared numerous times during instance lifetime. */
    EVP_PKEY *privateKey;

    /** Start of validity timespace of certificate. */
    QDateTime notBefore;
    /** End of validity timespace of certificate. */
    QDateTime notAfter;

    /** Key ID of certificate. */
    QString keyId;
    /** Key ID of issuer. */
    QString issuerKeyId;

    /** List of extensions of certificate. */
    QList< const X509Extension* > extensionList;
};

}

using namespace kCA_ossl;

Certificate::Certificate(QObject *parent) : QObject(parent), d(new CertificatePrivate)
{}

Certificate::Certificate(const Certificate &other, QObject *parent) :
    QObject(parent), d(new CertificatePrivate(other.d))
{}

Certificate::Certificate(const QString &filename, const QString &keyfile,
                         const QString &password, QObject *parent): QObject(parent), d(new CertificatePrivate)
{
  QFile source(filename);
  if (!source.exists()) {
    return;
  }
  source.open(QFile::ReadOnly);
  FILE *sourcefile = fdopen(source.handle(), "r");
  if (!sourcefile) {
    source.close();
    return;
  }
  d->internal = X509_new();
  d->internal = PEM_read_X509(sourcefile, NULL, NULL, 0);
  if (!d->internal) {
    rewind(sourcefile);
    d->internal = d2i_X509_fp(sourcefile, NULL);
  }
  fclose(sourcefile);
  source.close();

  // We could not read what we were given
  if (!d->internal) {
    return;
  }

  d->parseValues();
  d->valid = true;

  if (!loadPrivateKey(filename, password)) {
    loadPrivateKey(keyfile, password);
  }
}

Certificate::Certificate(X509 *other, const QByteArray &privateKey, QObject *parent) :
    QObject(parent),
    d(new CertificatePrivate)
{
  d->internal = X509_dup(other);
  if (d->internal) {
    d->parseValues();
    d->valid = true;

    if (privateKey.length() == 0) {
      return;
    }

    const unsigned char *keyData =(const unsigned char *) privateKey.data();
    d->privateKey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, &keyData, privateKey.length());
    if (d->privateKey) {
      if (!X509_check_private_key(d->internal, d->privateKey)) {
        EVP_PKEY_free(d->privateKey);
        d->privateKey = 0;
      }
    }
  }
}

Certificate::Certificate(const QByteArray certificate,
                         const QByteArray key, const QString password, QObject* parent) :
    QObject(parent),
    d(new CertificatePrivate)
{
  BIO *memory = BIO_new_mem_buf((void *) certificate.data(), certificate.length());
  PEM_read_bio_X509(memory, &d->internal, NULL, NULL);
  BIO_free(memory);

  if (d->internal) {
    d->parseValues();

    if (!key.isEmpty()) {
      this->addPrivateKey(key, password);
    }
    d->valid = true;
  }
}

Certificate::~Certificate()
{
  delete d;
}

bool Certificate::isValid() const
{
  return d->valid;
}

const X509Name Certificate::subject() const
{
  if (d->valid) {
    return X509Name(X509_get_subject_name(d->internal));
  }
  else {
    return X509Name();
  }
}

const X509Name Certificate::issuer() const
{
  if (d->valid) {
    return X509Name(X509_get_issuer_name(d->internal));
  }
  else {
    return X509Name();
  }
}

quint64 Certificate::serial() const
{
  return d->serial;
}

QByteArray Certificate::fingerprint() const
{
  unsigned int len;
  unsigned char md_value[EVP_MAX_MD_SIZE];
  if (!d->valid || !X509_digest(d->internal, EVP_sha1(), md_value, &len)) {
    return QByteArray();
  }

  return QByteArray::fromRawData((char *) md_value, len).toHex();
}

quint32 Certificate::subjectHash() const
{
  if (d->valid) {
    return X509_subject_name_hash(d->internal);
  }

  return 0;
}

quint32 Certificate::issuerHash() const
{
  if (d->valid) {
    return X509_issuer_name_hash(d->internal);
  }

  return 0;
}

bool Certificate::isCA() const
{
  return d->isCA;
}

QString &Certificate::keyId() const
{
  return d->keyId;
}

QString &Certificate::issuerKeyId() const
{
  return d->issuerKeyId;
}

QDateTime &Certificate::notBefore() const
{
  return d->notBefore;
}

QDateTime &Certificate::notAfter() const
{
  return d->notAfter;
}

Digest Certificate::digest() const
{
  if (!d->valid) {
    return kCA_ossl::SHA1;
  }

  int nid = OBJ_obj2nid(d->internal->sig_alg->algorithm);
  switch (nid) {
    case 119:
      return kCA_ossl::RIPEMD160;
      break;
    case 65:
      return kCA_ossl::SHA1;
      break;
    case 668:
      return kCA_ossl::SHA256;
      break;
    case 669:
      return kCA_ossl::SHA384;
      break;
    case 670:
      return kCA_ossl::SHA512;
      break;
  }

  return kCA_ossl::SHA1;
}

const QList< const X509Extension* > &Certificate::extensions() const
{
  return d->extensionList;
}

QByteArray Certificate::key(const QString password) const
{
  QByteArray result;

  if (!d->valid || !d->privateKey) {
    return result;
  }

  RSA *rsa = EVP_PKEY_get1_RSA(d->privateKey);
  if (!rsa) {
    return result;
  }

  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *buffer;

  if (memory) {
    if (!password.isEmpty())
      PEM_write_bio_RSAPrivateKey(memory, rsa, EVP_aes_256_cbc(), NULL, 0, 0,
                                  password.toUtf8().data());
    else
      PEM_write_bio_RSAPrivateKey(memory, rsa, NULL, NULL, 0, 0, NULL);

    BIO_get_mem_ptr(memory, &buffer);
    result = QByteArray(buffer->data, buffer->length);
    BIO_free(memory);
  }

  RSA_free(rsa);

  return result;
}

bool Certificate::loadPrivateKey(const QString& filename, const QString& password)
{
  if (filename.length() == 0) {
    return false;
  }

  QFile keyfile(filename);
  if (!keyfile.exists()) {
    return false;
  }

  keyfile.open(QFile::ReadOnly);
  bool result = addPrivateKey(keyfile.readAll(), password);
  keyfile.close();

  return result;
}

bool Certificate::addPrivateKey(const QByteArray &key, const QString &password)
{
  if (key.length() == 0) {
    return false;
  }

  if (!Utils::OSSL_ciphers_loaded) {
    OpenSSL_add_all_ciphers();
    Utils::OSSL_ciphers_loaded = true;
  }

  EVP_PKEY *pem;
  BIO *memory = BIO_new_mem_buf((void *) key.data(), key.length());

  // Try to decode as PEM first
  pem = PEM_read_bio_PrivateKey(memory, &d->privateKey, NULL,(void *) password.toUtf8().constData());

  if (!pem) {
    // PEM read failed, retry with DER encoding
    RSA *rsa = d2i_RSAPrivateKey_bio(memory, NULL);

    if (rsa) {
      d->privateKey = EVP_PKEY_new();
      if (d->privateKey) {
        EVP_PKEY_assign_RSA(d->privateKey, rsa);
      }
      RSA_free(rsa);
    }
  }
  BIO_free(memory);

  if (!d->privateKey) {
    return false;
  }

  // Check if private key matches certificate's public key
  EVP_PKEY *pubkey = X509_get_pubkey(d->internal);
  EVP_PKEY_copy_parameters(pubkey, d->privateKey);
  EVP_PKEY_free(pubkey);
  if (!X509_check_private_key(d->internal, d->privateKey)) {
    EVP_PKEY_free(d->privateKey);
    d->privateKey = 0;
    return false;
  }

  return true;
}

void Certificate::unloadPrivateKey()
{
  if (!d->privateKey) {
    return;
  }

  EVP_PKEY_free(d->privateKey);
  d->privateKey = 0;
}

X509 *Certificate::internal() const
{
  if (d->valid)
    return X509_dup(d->internal);
  else
    return 0;
}

const Certificate Certificate::sign(const Request &req, quint64 serial, const Digest digest,
                                    const QDateTime &notBefore, const QDateTime &notAfter,
                                    const QList< const X509Extension* > &extensions, QObject *parent) const
{
  if (!d->valid || !d->isCA || !d->privateKey || !req.isValid())
    return Certificate();

  if (notBefore < d->notBefore || notAfter > d->notAfter) {
    return Certificate();
  }

  X509 *newcert = X509_new();
  if (!Utils::build_X509(newcert, serial, req.subject().internal(),
        d->internal->cert_info->subject, X509_REQ_get_pubkey(req.internal()),
        notBefore, notAfter))
  {
    X509_free(newcert);
    return Certificate();
  }

  X509V3_CTX v3_context;
  X509V3_set_ctx_nodb(&v3_context);
  X509V3_set_ctx(&v3_context, d->internal, newcert, req.internal(), NULL, 0);
  foreach(const X509Extension *ext,
          (extensions.count() == 0 ? Utils::emailCertConstraints : extensions)) {
    X509_EXTENSION *x = X509V3_EXT_conf_nid(NULL, &v3_context, ext->getNid(),(char *) ext->value().constData());
    if (x) {
      X509_EXTENSION_set_critical(x, ext->isCritical());
      X509_add_ext(newcert, x, -1);
      X509_EXTENSION_free(x);
    }
  }

  int result = X509_sign(newcert, d->privateKey, Utils::load_digest(digest));
  if (result)
  {
    Certificate cert(newcert, parent);
    X509_free(newcert);
    return cert;
  }

  X509_free(newcert);
  return Certificate();
}

const Revocationlist Certificate::sign(const Revocationlist &crl, const Digest digest, QObject* parent) const
{
  if (!d->valid || !d->isCA || !d->privateKey) {
    return Revocationlist();
  }

  X509_CRL *newCrl = crl.internal();
  if (!newCrl) {
    return Revocationlist();
  }

  X509V3_CTX v3_context;
  X509_EXTENSION *extension;
  X509V3_set_ctx_nodb(&v3_context);
  X509V3_set_ctx(&v3_context, d->internal, NULL, NULL, newCrl, 0);
  extension = X509V3_EXT_conf_nid(NULL, &v3_context, NID_authority_key_identifier,
                                  (char *) "keyid:always,issuer:always");
  if (extension) {
    X509_CRL_add_ext(newCrl, extension, -1);
    X509_EXTENSION_free(extension);
  }

  // Set CRLs issuing timestamp to now.
  ASN1_TIME *ts = ASN1_TIME_new();
  if (ts) {
    ASN1_TIME_set(ts, QDateTime::currentDateTime().toUTC().toTime_t());
    X509_CRL_set_lastUpdate(newCrl, ts);
    ASN1_TIME_free(ts);
  }

  int result = X509_CRL_sign(newCrl, d->privateKey, Utils::load_digest(digest));
  if (result) {
    Revocationlist crl(newCrl, parent);
    X509_CRL_free(newCrl);
    return crl;
  }

  X509_CRL_free(newCrl);
  return Revocationlist();
}

QByteArray Certificate::toPem() const
{
  QByteArray result;

  if (!d->valid) {
    return result;
  }

  BIO *memory = BIO_new(BIO_s_mem());
  BUF_MEM *buffer;

  if (memory) {
    PEM_write_bio_X509(memory, d->internal);
    BIO_get_mem_ptr(memory, &buffer);
    result = QByteArray(buffer->data, buffer->length);
    BIO_free(memory);
  }

  return result;
}

bool Certificate::toPemFile(const QString &filename, WriteMode mode,
                            const QString& keyfile, const QString& password) const
{
  if (!d->valid) {
    return false;
  }

  QFile certfile(filename);
  if (!certfile.open(QFile::Append)) {
    return false;
  }

  bool result = false;
  QByteArray data = toPem();
  result = (certfile.write(data) == data.length());
  certfile.close();

  if (!result || !d->privateKey) {
    return result;
  }

  QString keyout = QString();
  switch (mode) {
    default:
    case NoPrivateKey:
      return result;
      break;
    case SameFile:
      keyout = filename;
      break;
    case OtherFile:
      keyout = keyfile;
      break;
  }

  QFile key(keyout);
  if (!key.open(QFile::Append)) {
    return false;
  }
  if (key.setPermissions(QFile::ReadOwner | QFile::WriteOwner)) {
    data = this->key(password);
    result = (key.write(data) == data.length());
  }
  key.close();

  return result;
}

bool Certificate::toDerFile(const QString& filename) const
{
  if (!d->valid) {
    return false;
  }

  QFile certfile(filename);
  if (!certfile.open(QFile::Append)) {
    return false;
  }

  bool result = false;

  BIO *b_out = BIO_new_fd(certfile.handle(), BIO_CLOSE);
  if (b_out) {
    if (i2d_X509_bio(b_out, d->internal)) {
      result = true;
    }
    BIO_free(b_out);
  }
  certfile.close();

  return result;
}

const QSslCertificate *Certificate::convert() const
{
  if (!d->valid) {
    return 0;
  }

#ifdef USE_QTNETWORK
  return new QSslCertificate(toPem(), QSsl::Pem);
#else  // USE_QTNETWORK
  return 0;
#endif // USE_QTNETWORK
}

Certificate& Certificate::operator=(const kCA_ossl::Certificate& other)
{
  *d = *(other.d);
  return *this;
}

Certificate Certificate::fromRawData(const QByteArray &certificate, const QByteArray &key, const QString &password)
{
  return Certificate(certificate, key, password);
}
